/* ------------------------------------------------------------------------------
  File: main.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Entry point for CHR-6d firmware
------------------------------------------------------------------------------ */ 

// Includes 
#include "stm32f10x.h"
#include "chr6d_startup.h"
#include "chr6d_config.h"
#include "chr6d_usart.h"
#include "chr6d_packet_handler.h"
#include "chr6d_FIR.h"

// Output of oversampler/decimator
short gDecimatedOutput[CHANNEL_COUNT];

// Intermediate buffer, stores result of oversampled, averaged data.  Used as input
// to FIR low-pass filter.
short gFIR_Input[CHANNEL_COUNT][MAX_INPUT_SIZE];

// Buffer to store data after all over-sampling, decimating, and filtering is performed.
short gFIR_Output[CHANNEL_COUNT];

/*******************************************************************************
* Function Name  : main
* Input          : None
* Output         : None
* Return         : None
* Description    : Entry point for CHR-6d firmware
*******************************************************************************/
int main(void)
{
	 unsigned int i;
	 unsigned int j;
	 int new_data;
	 
	 // Initialize the IMU clocks, ADC, DMA controller, GPIO pins, etc.
	 Initialize_IMU();
	 
	 // Fill gConfig structure from flash, or use default values if flash has not been initialized.
    GetConfiguration();
    
	 // Pre-initialize FIR input buffer to zero
	 for( i=0; i < CHANNEL_COUNT; i++ )
	 {
		  for( j=0; j < (MAX_INPUT_SIZE); j++ )
		  {
				gFIR_Input[i][j] = 0;
		  }
	 }
	 
	 // Start ADC1 Software Conversion.  ADC1 and ADC2 are configured to sample simultaneously, and to 
	 // use DMA1 to transfer data to memory.
    ADC_SoftwareStartConvCmd(ADC1, ENABLE);
	 
	 // Start TIM1 if in broadcast mode
	 if( gConfig.broadcast_enabled == MODE_BROADCAST )
	 {
		  EnableBroadcastMode( gConfig.broadcast_rate );
	 }
	 
	 // Main program loop
    while(1)
    {
		  // Filter new data if available (new_data = 0 if there was no new data ready, 1 otherwise)
		  new_data = process_input_buffers( gFIR_Input, gDecimatedOutput, gFIR_Output );
  
		  // Check if a packet has been received over the UART
		  if( gRXPacketReceived )
		  {
				ProcessPacket( (USARTPacket*)&gRXPacketBuffer[gRXPacketBufferStart] );

				// Increment RX packet buffer pointer.  The RX and TX buffer is circular.  The buffer is
				// empty if the "start" and "end" pointers point to the same location.
				gRXPacketBufferStart++;
				if( gRXPacketBufferStart >= RX_PACKET_BUFFER_SIZE )
				{
					 gRXPacketBufferStart = 0;
				}
				
				// If there are no more packets ready for processing, clear the gRXPacketReceived flag.
				// If there is another packet ready, leave the flag set so that the other RX packet will
				// be processed the next time through the main loop.
				if( gRXPacketBufferStart == gRXPacketBufferEnd )
				{
					 gRXPacketReceived = 0;
				}
		  }
    }
	 
}

/*******************************************************************************
* Function Name  : DMA1_Channel1_IRQHandler
* Input          : None
* Output         : None
* Return         : None
* Description    : Called in response to an interrupt from DMA1.  There are two
						 DMA triggered interrupts, one for a complete buffer transfer,
						 and one for a half buffer transfer.  The DMA buffer in memory
						 stores ADC data from the sensors, and is divided into two halves:
						 a "ping" buffer, and a "pong" buffer.  While one buffer is
						 filled, the data in the other is processed by the core.
*******************************************************************************/
void DMA1_Channel1_IRQHandler( void )
{

    // Interrupt service handler for half-transfer
    if( DMA_GetITStatus( DMA1_IT_HT1 ) != RESET )
    {
		  gPingBufferReady = 1;
		      
        DMA_ClearFlag( DMA1_FLAG_HT1 );
    }

    // Interrupt service handler for full transfer
    if( DMA_GetITStatus( DMA1_IT_TC1 ) != RESET )
    {
		  gPongBufferReady = 1;
        
        DMA_ClearFlag( DMA1_FLAG_TC1 );
    }

}

/*******************************************************************************
* Function Name  : TIM2_IRGHandler
* Input          : None
* Output         : None
* Return         : None
* Description    : Called in response to a interrupt from Timer2.  Timer 2 is
						 used to trigger transmission of SENSOR_DATA packets periodically.
						 While the sampling sampling frequency of the ADC converter
						 remains fixed, data is transmitted at variable rates based
						 on the timer configuration.
*******************************************************************************/
void TIM2_IRQHandler( void )
{
	 if( TIM_GetITStatus(TIM2, TIM_IT_Update ) != RESET )
	 {
		  
		  if( gConfig.broadcast_enabled == MODE_BROADCAST )
		  {
				// If the TX Packet Buffer is being used, then do not send data this time.  This is required since
				// the timer interrupt may have occured when the IMU was responding to a packet received over the
				// UART.
				if( TXPacketBufferReady )
				{
					 TXPacketBufferReady = 0;
					 SendDataPacket( );
					 TXPacketBufferReady = 1;
				}

		  }
		  
		  TIM_ClearITPendingBit(TIM2, TIM_IT_Update);	 
	 }
}

