/* ------------------------------------------------------------------------------
  File: chr6d_usart.h
  Author: CH Robotics
  Version: 1.0
  
  Description: Function declarations for USART communication
------------------------------------------------------------------------------ */ 
#include "stm32f10x.h"

#ifndef __CHR6D_USART_H
#define __CHR6D_USART_H

// Size of TX and RX buffers. These buffers hold raw transmitted and received data from the USART.
// These buffers should never have to be accessed directly by user code.
#define	RX_BUF_SIZE					40
#define 	TX_BUF_SIZE     			40

// Maximum number of data bytes that can be stored in a packet
#define	MAX_PACKET_DATA			20

// Sizes of buffers for storing RX and TX packet data.  The TX Packet Buffer is used
// to store packets that are "waiting" to be transmitted.  This is necessary since 
// multiple resources may need access to the USART hardware simultaneously.  The TX buffer
// acts as a moderator, storing packets and transmitting them when the hardware becomes available.
// The RX Packet Buffer stores packets received over the USART receiver.  Since multiple packets
// may arrive before a packet is processed, the RX Packet Buffer is neccessary to ensure that no data
// is lost.  Both the TX and RX Packet Buffers are circular.
#define	TX_PACKET_BUFFER_SIZE	10
#define	RX_PACKET_BUFFER_SIZE	10

// Definitions of states for USART receiver state machine (for receiving packets)
#define	USART_STATE_WAIT			1
#define	USART_STATE_TYPE			2
#define	USART_STATE_LENGTH		3
#define	USART_STATE_DATA			4
#define	USART_STATE_CHECKSUM		5

// Definitions of TX packet types (packets that are transmitted by the CHR-6d)
#define 	COMMAND_COMPLETE			0xB0
#define	COMMAND_FAILED				0xB1
#define	BAD_CHECKSUM				0xB2
#define	BAD_DATA_LENGTH			0xB3
#define	UNRECOGNIZED_PACKET		0xB4
#define	BUFFER_OVERFLOW			0xB5
#define	STATUS_REPORT				0xB6
#define	SENSOR_DATA					0xB7
#define	GYRO_BIAS_REPORT			0xB8
#define	ACCEL_BIAS_REPORT			0xB9
#define	FIR_CONFIG_REPORT			0xBA
#define	FIR_TAP_CONFIG_REPORT	0xBB
#define	ACTIVE_CHANNEL_REPORT	0xBC
#define	BROADCAST_MODE_REPORT	0xBD

// Definitions of RX packet types (packet that are received by the CHR-6d)
#define	SET_FIR_CORNERS			0x80
#define	SET_FIR_TAPS				0x81
#define	SET_ACTIVE_CHANNELS		0x82
#define	SET_SILENT_MODE			0x83
#define	SET_BROADCAST_MODE		0x84
#define	SET_X_GYRO_BIAS			0x85
#define	SET_Y_GYRO_BIAS			0x86
#define	SET_Z_GYRO_BIAS			0x87
#define	SET_X_ACCEL_BIAS			0x88
#define	SET_Y_ACCEL_BIAS			0x89
#define	SET_Z_ACCEL_BIAS			0x8A
#define	ZERO_RATE_GYROS			0x8B
#define	SELF_TEST					0x8C

#define	WRITE_TO_FLASH				0xA0

#define	GET_DATA						0x01
#define	GET_GYRO_BIAS				0x02
#define	GET_ACCEL_BIAS				0x03
#define	GET_FIR_CONFIG				0x04
#define	GET_FIR_TAP_CONFIG		0x05
#define	GET_ACTIVE_CHANNELS		0x06
#define	GET_BROADCAST_MODE		0x07

// Buffer, buffer index, and TX status flag for USART transmit
extern volatile char gTXBuf[TX_BUF_SIZE];
extern volatile int gTXBufPtr;
extern volatile char gTXBusy;

// USART RX buffer and associated index and flags
extern volatile char gRXBuf[RX_BUF_SIZE];
extern volatile int gRXBufPtr;
extern volatile char gRXPacketReceived;
extern volatile char gRXBufOverrun;
	 
	 
// Structure for storing TX and RX packet data
typedef struct __USARTPacket
{
	 uint8_t PT;				// Packet type
	 uint8_t length;			// Number of bytes in data section
	 uint16_t checksum;		// Checksum
	 
	 uint8_t packet_data[MAX_PACKET_DATA];
} USARTPacket;

// USART interface functions
int USART1_transmit( char* txdata, int length );
int TXBufPush( char txdata );
char TXBufPop( void );
void USART1_TX_start( void );

void USART1_IRQHandler(void);

// Function for copying next packet in the TXPacketBuffer into the TX buffer.
void SendNextPacket( void );
void AddTXPacket( USARTPacket* new_packet );
void AddRXPacket( USARTPacket* new_packet );

uint16_t ComputeChecksum( USARTPacket* new_packet );

void SendTXPacket( USARTPacket* new_packet );
void SendTXPacketSafe( USARTPacket* new_packet );

// Global queue for packets to be transmitted when the UART transmitter becomes available.
extern volatile USARTPacket gTXPacketBuffer[TX_PACKET_BUFFER_SIZE];
extern volatile uint8_t gTXPacketBufferStart;
extern volatile uint8_t gTXPacketBufferEnd;

extern volatile char TXPacketBufferReady;
extern volatile char gCopyingTXPacketToBuffer;

// Global queue for storing received packets
extern volatile USARTPacket gRXPacketBuffer[TX_PACKET_BUFFER_SIZE];
extern volatile uint8_t gRXPacketBufferStart;
extern volatile uint8_t gRXPacketBufferEnd;


#endif