/* ------------------------------------------------------------------------------
  File: chr6d_packet_handler.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Functions used to handle packets sent and received by the CHR-6d
------------------------------------------------------------------------------ */ 

#include "stm32f10x.h"
#include "chr6d_packet_handler.h"
#include "chr6d_config.h"
#include "chr6d_FIR.h"

/*******************************************************************************
* Function Name  : ProcessPacket
* Input          : USARTPacket*
* Output         : None
* Return         : None
* Description    : Takes a packet received over the UART and processes it, calling
*						the appropriate functions to act on the new data.
*******************************************************************************/
void ProcessPacket( USARTPacket* new_packet )
{
	 USARTPacket response_packet;
	 int result;
	 
	 // Determine the packet type
	 switch( new_packet->PT )
	 {
		  // -----------------------------------------------------------------------------
		  // SET_FIR_CORNERS packet - Sets the 3-dB corner frequency of each channel.
		  // Corner freq. = (x-1)*10 Hz.  If x = 0, or x = 1, then the FIR filter
		  // is disabled for that channel.
		  // -----------------------------------------------------------------------------
		  case SET_FIR_CORNERS:
				gConfig.x_accel_corner = (uint8_t)new_packet->packet_data[5];
				if( gConfig.x_accel_corner < 2 ) {
					 gConfig.x_accel_corner = FIR_DISABLED;
				}
				
				gConfig.y_accel_corner = (uint8_t)new_packet->packet_data[4];
				if( gConfig.y_accel_corner < 2 ) {
					 gConfig.y_accel_corner = FIR_DISABLED;
				}
				
				gConfig.z_accel_corner = (uint8_t)new_packet->packet_data[3];
				if( gConfig.z_accel_corner < 2 ) {
					 gConfig.z_accel_corner = FIR_DISABLED;
				}
				
				gConfig.x_gyro_corner = (uint8_t)new_packet->packet_data[2];
				if( gConfig.x_gyro_corner < 2 ) {
					 gConfig.x_gyro_corner = FIR_DISABLED;
				}
				
				gConfig.y_gyro_corner = (uint8_t)new_packet->packet_data[1];
				if( gConfig.y_gyro_corner < 2 ) {
					 gConfig.y_gyro_corner = FIR_DISABLED;
				}
				
				gConfig.z_gyro_corner = (uint8_t)new_packet->packet_data[0];
				if( gConfig.z_gyro_corner < 2 ) {
					 gConfig.z_gyro_corner = FIR_DISABLED;
				}
				
				SendCommandSuccessPacket( SET_FIR_CORNERS );				
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_FIR_TAPS packet - Sets the number of taps used in each FIR filter.
		  // 0 = 8 taps, 1, = 16 taps, 2 = 32 taps, 3 = 64 taps
		  // -----------------------------------------------------------------------------
		  case SET_FIR_TAPS:
				gConfig.x_accel_taps = (uint8_t)(new_packet->packet_data[1] & 0x03);
				gConfig.y_accel_taps = (uint8_t)((new_packet->packet_data[1] >> 2) & 0x03);
				gConfig.z_accel_taps	= (uint8_t)((new_packet->packet_data[1] >> 4) & 0x03);
				gConfig.x_gyro_taps	= (uint8_t)((new_packet->packet_data[1] >> 6) & 0x03);
				gConfig.y_gyro_taps	= (uint8_t)((new_packet->packet_data[0]) & 0x03);
				gConfig.z_gyro_taps	= (uint8_t)((new_packet->packet_data[0] >> 2) & 0x03);
		  
				SendCommandSuccessPacket( SET_FIR_TAPS );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_ACTIVE_CHANNELS packet - Sets channels whose data will be transmitted
		  // when a GET_DATA packet is sent, or when Broadcast Mode is enabled
		  // -----------------------------------------------------------------------------
		  case SET_ACTIVE_CHANNELS:
				gConfig.x_accel_enabled = (uint8_t)( (new_packet->packet_data[0]) & 0x01);
				gConfig.y_accel_enabled = (uint8_t)( (new_packet->packet_data[0] >> 1) & 0x01);
				gConfig.z_accel_enabled = (uint8_t)( (new_packet->packet_data[0] >> 2) & 0x01);
				gConfig.x_gyro_enabled = (uint8_t)( (new_packet->packet_data[0] >> 3) & 0x01);
				gConfig.y_gyro_enabled = (uint8_t)( (new_packet->packet_data[0] >> 4) & 0x01);
				gConfig.z_gyro_enabled = (uint8_t)( (new_packet->packet_data[0] >> 5) & 0x01);
		  
				SendCommandSuccessPacket( SET_ACTIVE_CHANNELS );
				
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_SILENT_MODE packet - Turns off broadcast mode.
		  // -----------------------------------------------------------------------------
		  case SET_SILENT_MODE:
				DisableBroadcastMode( );
		  
				SendCommandSuccessPacket( SET_SILENT_MODE );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_BROADCAST_MODE packet - turns on broadcast mode with the frequency
		  // specified in packet_data[0].
		  // -----------------------------------------------------------------------------
		  case SET_BROADCAST_MODE:
				EnableBroadcastMode( new_packet->packet_data[0] );
		  
				SendCommandSuccessPacket( SET_BROADCAST_MODE );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_X_GYRO_BIAS packet - Sets bias term for x-axis rate gyro.
		  // -----------------------------------------------------------------------------
		  case SET_X_GYRO_BIAS:
				gConfig.x_gyro_bias = ((uint16_t)new_packet->packet_data[0] << 8) | ((uint16_t)new_packet->packet_data[1] & 0x0FF);
		  
				SendCommandSuccessPacket( SET_X_GYRO_BIAS );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_Y_GYRO_BIAS packet - Sets bias term for y-axis rate gyro.
		  // -----------------------------------------------------------------------------
		  case SET_Y_GYRO_BIAS:
				gConfig.y_gyro_bias = ((uint16_t)new_packet->packet_data[0] << 8) | ((uint16_t)new_packet->packet_data[1] & 0x0FF);
		  
				SendCommandSuccessPacket( SET_Y_GYRO_BIAS );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_Z_GYRO_BIAS packet - Sets bias term for z-axis rate gyro.
		  // -----------------------------------------------------------------------------
		  case SET_Z_GYRO_BIAS:
				gConfig.z_gyro_bias = ((uint16_t)new_packet->packet_data[0] << 8) | ((uint16_t)new_packet->packet_data[1] & 0x0FF);
		  
				SendCommandSuccessPacket( SET_Z_GYRO_BIAS );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_X_ACCEL_BIAS packet - Sets bias term for x-axis accelerometer
		  // -----------------------------------------------------------------------------
		  case SET_X_ACCEL_BIAS:
				gConfig.x_accel_bias = (((uint16_t)new_packet->packet_data[0] << 8) & 0x0FF00) | ((uint16_t)new_packet->packet_data[1] & 0x0FF);
		  
				SendCommandSuccessPacket( SET_X_ACCEL_BIAS );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_Y_ACCEL_BIAS packet - Sets bias term for y-axis accelerometer
		  // -----------------------------------------------------------------------------
		  case SET_Y_ACCEL_BIAS:
				gConfig.y_accel_bias = ((uint16_t)new_packet->packet_data[0] << 8) | ((uint16_t)new_packet->packet_data[1] & 0x0FF);
		  
				SendCommandSuccessPacket( SET_Y_ACCEL_BIAS );
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SET_Z_ACCEL_BIAS packet - Sets bias term for z-axis accelerometer
		  // -----------------------------------------------------------------------------
		  case SET_Z_ACCEL_BIAS:
				gConfig.z_accel_bias = ((uint16_t)new_packet->packet_data[0] << 8) | ((uint16_t)new_packet->packet_data[1] & 0x0FF);
		  
				SendCommandSuccessPacket( SET_Z_ACCEL_BIAS );
		  break;

		  // -----------------------------------------------------------------------------
		  // ZERO_RATE_GYROS packet - starts gyro zero command
		  // -----------------------------------------------------------------------------
		  case ZERO_RATE_GYROS:
				StartGyroCalibration();
		  
				// No COMMAND_COMPLETE packet is sent from here.  When the gyro zeroing is finished,
				// a COMMAND_COMPLETE packet is sent from within the filtering code.
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // SELF_TEST packet - starts automatic self-test command
		  // -----------------------------------------------------------------------------
		  case SELF_TEST:
				StartSelfTest();
		  break;
		  
		  // -----------------------------------------------------------------------------
		  // WRITE_TO_FLASH packet - saves all settings to flash.
		  // -----------------------------------------------------------------------------
		  case WRITE_TO_FLASH:
				
				// Disable the ADC temporarily - the WRITE_TO_FLASH command takes longer to complete
				// than it takes to fill an input buffer with data.
				ADC_SoftwareStartConvCmd(ADC1, DISABLE);
				
				// If BROADCAST_MODE is enabled, turn it off while writing configuration to flash
				if( gConfig.broadcast_enabled == MODE_BROADCAST )
				{
					 // Disable Timer 2
					 TIM_Cmd( TIM2, DISABLE );	
		  
					 result = WriteConfigurationToFlash( );
					 
					 // Enable Timer 2
					 TIM_Cmd( TIM2, ENABLE );	
				}
				else
				{
					 result = WriteConfigurationToFlash( );
				}
		  
				ADC_SoftwareStartConvCmd(ADC1, ENABLE);
								
				if( result == FLASH_COMPLETE )
				{
					 SendCommandSuccessPacket( WRITE_TO_FLASH );
				}
				else
				{
					 SendCommandFailedPacket( WRITE_TO_FLASH, result );
				}
				
		  break;
		  
		  // ------------------------------------------------------------------------------
		  // GET_DATA packet - ignored if IMU is in broadcast mode.  If in listen mode,
		  // return data from all active channels.
		  // ------------------------------------------------------------------------------
		  case GET_DATA:
				if( !gConfig.broadcast_enabled )
				{
					 SendDataPacket();
				}
		  break;
		  
		  // ------------------------------------------------------------------------------
		  // GET_GYRO_BIAS packet - Sends a packet containing the bias values
		  // for each gyro channel.
		  // ------------------------------------------------------------------------------
		  case GET_GYRO_BIAS:
				SendGyroBiasPacket();
		  break;
		  
		  // ------------------------------------------------------------------------------
		  // GET_ACCEL_BIAS packet - Sends a packet containing the bias values
		  // for each accelerometer annel.
		  // ------------------------------------------------------------------------------
		  case GET_ACCEL_BIAS:
				SendAccelBiasPacket();
		  break;
		  
		  // ------------------------------------------------------------------------------
		  // GET_FIR_CONFIG packet - Sends a packet describing the corner frequency
		  // configuration for each individual channel's FIR filter
		  // ------------------------------------------------------------------------------
		  case GET_FIR_CONFIG:
				SendFIR_Packet();
		  break;
		  
		  // ------------------------------------------------------------------------------
		  // GET_FIR_TAP_CONFIG packet - Returns a packet containing the number of taps
		  // used in each sensor's FIR filter.
		  // ------------------------------------------------------------------------------
		  case GET_FIR_TAP_CONFIG:
				SendFIR_TapPacket();
		  break;
		  
		  // ------------------------------------------------------------------------------
		  // GET_ACTIVE_CHANNELS packet - Returns a packet specifying whether each sensor
		  // channel is active or inactive.
		  // ------------------------------------------------------------------------------
		  case GET_ACTIVE_CHANNELS:
				SendActiveChannelPacket();
		  break;
		  
		  case GET_BROADCAST_MODE:
				SendTransmitModePacket();
		  break;
		  
		  default:
				// The packet was not recognized.  Send an UNRECOGNIZED_PACKET packet
				response_packet.PT = UNRECOGNIZED_PACKET;
				response_packet.length = 1;
				response_packet.packet_data[0] = new_packet->PT;
				response_packet.checksum = ComputeChecksum( &response_packet );
				
				SendTXPacketSafe( &response_packet );
		  
		  break;
	 }
	 
}


/*******************************************************************************
* Function Name  : SendCommandSuccessPacket
* Input          : int
* Output         : None
* Return         : None
* Description    : Sends a COMMAND_COMPLETE packet over the UART transmitter.
*******************************************************************************/
void SendCommandSuccessPacket( int command )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = COMMAND_COMPLETE;
	 NewPacket.length = 1;
	 NewPacket.packet_data[0] = command;
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}


/*******************************************************************************
* Function Name  : SendCommandFailedPacket
* Input          : int
* Output         : None
* Return         : None
* Description    : Sens a COMMAND_FAILED packet over the UART transmitter
*******************************************************************************/
void SendCommandFailedPacket( int command, int flag  )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = COMMAND_FAILED;
	 NewPacket.length = 2;
	 NewPacket.packet_data[0] = (char)command;
	 NewPacket.packet_data[1] = (char)flag;
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}


/*******************************************************************************
* Function Name  : SendDataPacket
* Input          : Uses global data buffers defined in main.c, and a global configuration
						  structure defined in chr6d_config.c
* Output         : None
* Return         : None
* Description    : Sends a packet containing the most recent data from all the 
						  active IMU channels
*******************************************************************************/
void SendDataPacket( void )
{
	 USARTPacket NewPacket;
	 int active_channels = 0;
	  
	 NewPacket.PT = SENSOR_DATA;
	 
	 if( gConfig.z_gyro_enabled )
	 {
		  NewPacket.packet_data[2*active_channels+1] = (uint8_t)((gFIR_Output[5] >> 8) & 0x0FF);
		  NewPacket.packet_data[2*active_channels+2] = (uint8_t)((gFIR_Output[5]) & 0x0FF);
		  
		  active_channels++;
	 }
	 
	 if( gConfig.y_gyro_enabled )
	 {
		  NewPacket.packet_data[2*active_channels+1] = (uint8_t)((gFIR_Output[3] >> 8) & 0x0FF);
		  NewPacket.packet_data[2*active_channels+2] = (uint8_t)((gFIR_Output[3]) & 0x0FF);
		  
		  active_channels++;		  
	 }
	 
	 if( gConfig.x_gyro_enabled )
	 {
		  NewPacket.packet_data[2*active_channels+1] = (uint8_t)((gFIR_Output[4] >> 8) & 0x0FF);
		  NewPacket.packet_data[2*active_channels+2] = (uint8_t)((gFIR_Output[4]) & 0x0FF);
		  
		  active_channels++;
	 }
	 
	 if( gConfig.z_accel_enabled )
	 {
		  NewPacket.packet_data[2*active_channels+1] = (uint8_t)((gFIR_Output[2] >> 8) & 0x0FF);
		  NewPacket.packet_data[2*active_channels+2] = (uint8_t)((gFIR_Output[2]) & 0x0FF);
		  
		  active_channels++;		  
	 }
	 
	 if( gConfig.y_accel_enabled )
	 {
		  NewPacket.packet_data[2*active_channels+1] = (uint8_t)((gFIR_Output[1] >> 8) & 0x0FF);
		  NewPacket.packet_data[2*active_channels+2] = (uint8_t)((gFIR_Output[1]) & 0x0FF);
		  
		  active_channels++;
	 }
	 
	 if( gConfig.x_accel_enabled )
	 {
		  NewPacket.packet_data[2*active_channels+1] = (uint8_t)((gFIR_Output[0] >> 8) & 0x0FF);
		  NewPacket.packet_data[2*active_channels+2] = (uint8_t)((gFIR_Output[0]) & 0x0FF);
		  
		  active_channels++;
	 }
	 
	 NewPacket.length = 2*active_channels + 1;
	 
	 NewPacket.packet_data[0] = (gConfig.z_gyro_enabled << 5) | (gConfig.y_gyro_enabled << 4) | (gConfig.x_gyro_enabled << 3);
	 NewPacket.packet_data[0] |= (gConfig.z_accel_enabled << 2) | (gConfig.y_accel_enabled << 1) | (gConfig.x_accel_enabled);
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 // This is the ONLY packet that is not required to send packets using the SendTXPacketSafe() function call.
	 SendTXPacket( &NewPacket );
}

/*******************************************************************************
* Function Name  : SendGyroBiasPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a GYRO_BIAS_REPORT packet
*******************************************************************************/
void SendGyroBiasPacket( )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = GYRO_BIAS_REPORT;
	 NewPacket.length = 6;
	 
	 // X gyro bias
	 NewPacket.packet_data[5] = (uint8_t)(gConfig.x_gyro_bias & 0x0FF);
	 NewPacket.packet_data[4] = (uint8_t)((gConfig.x_gyro_bias >> 8) & 0x0FF);
	 
	 // Y gyro bias
	 NewPacket.packet_data[3] = (uint8_t)(gConfig.y_gyro_bias & 0x0FF);
	 NewPacket.packet_data[2] = (uint8_t)((gConfig.y_gyro_bias >> 8) & 0x0FF);
	 
	 // Z gyro bias
	 NewPacket.packet_data[1] = (uint8_t)(gConfig.z_gyro_bias & 0x0FF);
	 NewPacket.packet_data[0] = (uint8_t)((gConfig.z_gyro_bias >> 8) & 0x0FF);
	 
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}


/*******************************************************************************
* Function Name  : SendAccelBiasPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a GYRO_BIAS_REPORT packet
*******************************************************************************/
void SendAccelBiasPacket( )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = ACCEL_BIAS_REPORT;
	 NewPacket.length = 6;
	 
	 // X gyro bias
	 NewPacket.packet_data[5] = (uint8_t)(gConfig.x_accel_bias & 0x0FF);
	 NewPacket.packet_data[4] = (uint8_t)((gConfig.x_accel_bias >> 8) & 0x0FF);
	 
	 // Y gyro bias
	 NewPacket.packet_data[3] = (uint8_t)(gConfig.y_accel_bias & 0x0FF);
	 NewPacket.packet_data[2] = (uint8_t)((gConfig.y_accel_bias >> 8) & 0x0FF);
	 
	 // Z gyro bias
	 NewPacket.packet_data[1] = (uint8_t)(gConfig.z_accel_bias & 0x0FF);
	 NewPacket.packet_data[0] = (uint8_t)((gConfig.z_accel_bias >> 8) & 0x0FF);
	 
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}

/*******************************************************************************
* Function Name  : SendFIR_Packet
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a FIR_CONFIG_REPORT packet
*******************************************************************************/
void SendFIR_Packet( )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = FIR_CONFIG_REPORT;
	 NewPacket.length = 3;
	 
	 NewPacket.packet_data[2] = (gConfig.x_accel_corner & 0x0F) | ((gConfig.y_accel_corner << 4) & 0xF0);
	 NewPacket.packet_data[1] = (gConfig.z_accel_corner & 0x0F) | ((gConfig.x_gyro_corner << 4) & 0xF0);
	 NewPacket.packet_data[0] = (gConfig.y_gyro_corner & 0x0F) | ((gConfig.z_gyro_corner << 4) & 0xF0);
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}

/*******************************************************************************
* Function Name  : SendFIR_TapPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a FIR_TAP_CONFIG_REPORT packet
*******************************************************************************/
void SendFIR_TapPacket( )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = FIR_TAP_CONFIG_REPORT;
	 NewPacket.length = 2;
	 
	 NewPacket.packet_data[1] = ((gConfig.x_accel_taps) & 0x03) | ((gConfig.y_accel_taps & 0x03) << 2 ) | ((gConfig.z_accel_taps & 0x03 ) << 4) | ((gConfig.x_gyro_taps & 0x03) << 6);
	 NewPacket.packet_data[0] = ((gConfig.y_gyro_taps) & 0x03) | ((gConfig.z_gyro_taps & 0x03) << 2 );
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}

/*******************************************************************************
* Function Name  : SendActiveChannelPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a ACTIVE_CHANNEL_REPORT packet
*******************************************************************************/
void SendActiveChannelPacket( )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = ACTIVE_CHANNEL_REPORT;
	 NewPacket.length = 1;
	 
	 NewPacket.packet_data[0] = ((gConfig.x_accel_enabled) & 0x01) | ((gConfig.y_accel_enabled << 1) & 0x02) | ((gConfig.z_accel_enabled << 2) & 0x04)
										  | ((gConfig.x_gyro_enabled << 3) & 0x08) | ((gConfig.y_gyro_enabled << 4) & 0x10) | ((gConfig.z_gyro_enabled << 5) & 0x20);
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}


/*******************************************************************************
* Function Name  : SendTransmitModePacket
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a TRANSMIT_MODE_REPORT packet
*******************************************************************************/
void SendTransmitModePacket( )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = BROADCAST_MODE_REPORT;
	 NewPacket.length = 2;
	 
	 NewPacket.packet_data[0] = (gConfig.broadcast_rate & 0x0FF);
	 NewPacket.packet_data[1] = (gConfig.broadcast_enabled & 0x01);
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}

/*******************************************************************************
* Function Name  : SendStatusReportPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : Sends a STATUS_REPORT packet.  Sent after self-test has been run.
*******************************************************************************/
void SendStatusReportPacket( uint16_t result )
{
	 USARTPacket NewPacket;
	 
	 NewPacket.PT = STATUS_REPORT;
	 NewPacket.length = 1;
	 
	 NewPacket.packet_data[0] = (result & 0x0FF);
	 
	 NewPacket.checksum = ComputeChecksum( &NewPacket );
	 
	 SendTXPacketSafe( &NewPacket );
}
