/* ------------------------------------------------------------------------------
  File: chr6d_config.f
  Author: CH Robotics
  Version: 1.0
  
  Description: Preprocessor definitions and func declarations for IMU configuration
------------------------------------------------------------------------------ */ 
#ifndef __CHR6D_CONFIG_H
#define __CHR6D_CONFIG_H


typedef struct __CHR6d_config 
{
	 // Gyro and accelerometer default biases
	 uint16_t x_accel_bias;
	 uint16_t y_accel_bias;
	 uint16_t z_accel_bias;
	 uint16_t x_gyro_bias;
	 uint16_t y_gyro_bias;
	 uint16_t z_gyro_bias;
	 
	 // Gyro and accelerometer default FIR corner frequencies
	 uint8_t x_accel_corner;
	 uint8_t y_accel_corner;
	 uint8_t z_accel_corner;
	 uint8_t x_gyro_corner;
	 uint8_t y_gyro_corner;
	 uint8_t z_gyro_corner;
	 
	 // Gyro and accelerometer default FIR filter tap lengths
	 uint8_t x_accel_taps;
	 uint8_t y_accel_taps;
	 uint8_t z_accel_taps;
	 uint8_t x_gyro_taps;
	 uint8_t y_gyro_taps;
	 uint8_t z_gyro_taps;
	 
	 // Chanel enable/disable flags
	 uint8_t x_accel_enabled;
	 uint8_t y_accel_enabled;
	 uint8_t z_accel_enabled;
	 uint8_t x_gyro_enabled;
	 uint8_t y_gyro_enabled;
	 uint8_t z_gyro_enabled;
	 
	 uint8_t broadcast_enabled;
	 uint8_t broadcast_rate;
	 
} CHR6d_config;

extern CHR6d_config gConfig;

void GetConfiguration( void );
int WriteConfigurationToFlash( void );

void UpdateBroadcastRate( uint8_t new_rate );
void EnableBroadcastMode( uint8_t new_rate );
void DisableBroadcastMode( void );
void StartGyroCalibration( void );
void StopGyroCalibration( void );

void StartSelfTest( void );
void StopSelfTest( uint16_t result );

// Addresses of configuration registers.  On the CHR6d, the addresses below
// place the configuration registers into flash memory at page 32.  The addresses
// are fairly arbitrary, and can be anything so long as they do not interfere with
// program memory.  If custom code brings the program size to above 32 K, these addresses
// should be increased.
#define		OFFSET_CONF1_ADDR			(uint32_t)0x08008000
#define		OFFSET_CONF2_ADDR			(uint32_t)0x08008004
#define		OFFSET_CONF3_ADDR			(uint32_t)0x08008008
#define	 	FIR_CONF_ADDR				(uint32_t)0x0800800C
#define		FIR_TAP_CONF_ADDR			(uint32_t)0x08008010
#define		USART_CONF_ADDR			(uint32_t)0x08008014

// Preprocessor definitions used to simplify access to parameters stored in FLASH
#define		GET_ACCEL_X_OFFSET()		(uint16_t)( (*(__IO uint32_t*)(OFFSET_CONF1_ADDR)) & 0x0FFFF )
#define		GET_ACCEL_Y_OFFSET()		(uint16_t)( ((*(__IO uint32_t*)(OFFSET_CONF1_ADDR)) >> 16) & 0x0FFFF )
#define		GET_ACCEL_Z_OFFSET()		(uint16_t)( (*(__IO uint32_t*)(OFFSET_CONF2_ADDR)) & 0x0FFFF )
#define		GET_GYRO_X_OFFSET()		(uint16_t)( ((*(__IO uint32_t*)(OFFSET_CONF2_ADDR)) >> 16) & 0x0FFFF )
#define		GET_GYRO_Y_OFFSET()		(uint16_t)( (*(__IO uint32_t*)(OFFSET_CONF3_ADDR)) & 0x0FFFF )
#define		GET_GYRO_Z_OFFSET()		(uint16_t)( ((*(__IO uint32_t*)(OFFSET_CONF3_ADDR)) >> 16) & 0x0FFFF )

#define		FIR_DISABLED				0
#define		FIR_CORNER_10HZ			2
#define		FIR_CORNER_20HZ			3
#define		FIR_CORNER_30HZ			4
#define		FIR_CORNER_40HZ			5
#define		FIR_CORNER_50HZ			6
#define		FIR_CORNER_60HZ			7
#define		FIR_CORNER_70HZ			8
#define		FIR_CORNER_80HZ			9
#define		FIR_CORNER_90HZ			10
#define		FIR_CORNER_100HZ			11
#define		FIR_CORNER_110HZ			12
#define		FIR_CORNER_120HZ			13
#define		FIR_CORNER_130HZ			14
#define		FIR_CORNER_140HZ			15

#define		GET_ACCEL_X_CORNER()		(uint8_t)( (*(__IO uint32_t*)(FIR_CONF_ADDR)) & 0x0F )
#define		GET_ACCEL_Y_CORNER()		(uint8_t)( ((*(__IO uint32_t*)(FIR_CONF_ADDR)) >> 4) & 0x0F )
#define		GET_ACCEL_Z_CORNER()		(uint8_t)( ((*(__IO uint32_t*)(FIR_CONF_ADDR)) >> 8) & 0x0F )

#define		GET_GYRO_X_CORNER()		(uint8_t)( ((*(__IO uint32_t*)(FIR_CONF_ADDR)) >> 12)  & 0x0F )
#define		GET_GYRO_Y_CORNER()		(uint8_t)( ((*(__IO uint32_t*)(FIR_CONF_ADDR)) >> 16) & 0x0F )
#define		GET_GYRO_Z_CORNER()		(uint8_t)( ((*(__IO uint32_t*)(FIR_CONF_ADDR)) >> 20) & 0x0F )

#define		FIR_TAPS_8					0
#define		FIR_TAPS_16					1
#define		FIR_TAPS_32					2
#define		FIR_TAPS_64					3

#define		GET_ACCEL_X_TAPS()		(uint8_t)( (*(__IO uint32_t*)(FIR_TAP_CONF_ADDR)) & 0x03 )
#define		GET_ACCEL_Y_TAPS()		(uint8_t)( ((*(__IO uint32_t*)(FIR_TAP_CONF_ADDR)) >> 2) & 0x03 )
#define		GET_ACCEL_Z_TAPS()		(uint8_t)( ((*(__IO uint32_t*)(FIR_TAP_CONF_ADDR)) >> 4) & 0x03 )

#define		GET_GYRO_X_TAPS()			(uint8_t)( ((*(__IO uint32_t*)(FIR_TAP_CONF_ADDR)) >> 6) & 0x03 )
#define		GET_GYRO_Y_TAPS()			(uint8_t)( ((*(__IO uint32_t*)(FIR_TAP_CONF_ADDR)) >> 8) & 0x03 )
#define		GET_GYRO_Z_TAPS()			(uint8_t)( ((*(__IO uint32_t*)(FIR_TAP_CONF_ADDR)) >> 10) & 0x03 )

#define		CHANNEL_ENABLED			1
#define		CHANNEL_DISABLED			0

#define		IS_ACCEL_X_ENABLED()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) ) & 0x01 )
#define		IS_ACCEL_Y_ENABLED()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 1) & 0x01 )
#define		IS_ACCEL_Z_ENABLED()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 2) & 0x01 )
#define		IS_GYRO_X_ENABLED()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 3) & 0x01 )
#define		IS_GYRO_Y_ENABLED()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 4) & 0x01 )
#define		IS_GYRO_Z_ENABLED()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 5) & 0x01 )

#define		MODE_BROADCAST				1
#define		MODE_LISTEN					0

#define		GET_TRANSMIT_MODE()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 6) & 0x01 )
#define		GET_BROADCAST_RATE()		(uint8_t)( ((*(__IO uint32_t*)(USART_CONF_ADDR)) >> 8) & 0x00FF )


#endif