/* ------------------------------------------------------------------------------
  File: chr6d_config.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Function definitions for IMU configuration
------------------------------------------------------------------------------ */ 
#include "stm32f10x.h"
#include "chr6d_config.h"
#include "chr6d_FIR.h"

// Global structure storing IMU configuration parameters
CHR6d_config gConfig;

/*******************************************************************************
* Function Name  : GetConfiguration
* Input          : None
* Output         : None
* Return         : None
* Description    : Fills the gConfig structure with IMU configuration data.
						  If configuration data has been written to flash, then load it.
						  If not, then use factory defaults.
*******************************************************************************/
void GetConfiguration( void )
{
	 // If flash has not been programmed yet, then use default configuration.  Otherwise, load configuration from flash
	 if( GET_ACCEL_X_OFFSET() == 0xFFFF )
	 {
		  // Gyro and accelerometer default biases
		  gConfig.x_accel_bias = 0x802F;
		  gConfig.y_accel_bias = 0x802F;
		  gConfig.z_accel_bias = 0x802F;
		  gConfig.x_gyro_bias = 0x5F6B;
		  gConfig.y_gyro_bias = 0x5F6B;
		  gConfig.z_gyro_bias = 0x5F6B;
		  
		  // Gyro and accelerometer default FIR corner frequencies
		  gConfig.x_accel_corner = FIR_CORNER_140HZ;
		  gConfig.y_accel_corner = FIR_CORNER_140HZ;
		  gConfig.z_accel_corner = FIR_CORNER_140HZ;
		  gConfig.x_gyro_corner = FIR_CORNER_140HZ;
		  gConfig.y_gyro_corner = FIR_CORNER_140HZ;
		  gConfig.z_gyro_corner = FIR_CORNER_140HZ;
		  
		  // Gyro and accelerometer default FIR filter tap lengths
		  gConfig.x_accel_taps = FIR_TAPS_32;
		  gConfig.y_accel_taps = FIR_TAPS_32;
		  gConfig.z_accel_taps = FIR_TAPS_32;
		  gConfig.x_gyro_taps = FIR_TAPS_32;
		  gConfig.y_gyro_taps = FIR_TAPS_32;
		  gConfig.z_gyro_taps = FIR_TAPS_32;
		  
		  // Channel enable/disable flags
 		  gConfig.x_accel_enabled = CHANNEL_ENABLED;
		  gConfig.y_accel_enabled = CHANNEL_ENABLED;
		  gConfig.z_accel_enabled = CHANNEL_ENABLED;
		  gConfig.x_gyro_enabled = CHANNEL_ENABLED;
		  gConfig.y_gyro_enabled = CHANNEL_ENABLED;
		  gConfig.z_gyro_enabled = CHANNEL_ENABLED;

		  // Broadcast mode or listen mode
		  gConfig.broadcast_enabled = MODE_BROADCAST;
		  gConfig.broadcast_rate = 237;
	 }
	 else
	 {
	  
		  // Gyro and accelerometer default biases
		  gConfig.x_accel_bias = GET_ACCEL_X_OFFSET();
		  gConfig.y_accel_bias = GET_ACCEL_Y_OFFSET();
		  gConfig.z_accel_bias = GET_ACCEL_Z_OFFSET();
		  gConfig.x_gyro_bias = GET_GYRO_X_OFFSET();
		  gConfig.y_gyro_bias = GET_GYRO_Y_OFFSET();
		  gConfig.z_gyro_bias = GET_GYRO_Z_OFFSET();
		  
		  // Gyro and accelerometer default FIR corner frequencies
		  gConfig.x_accel_corner = GET_ACCEL_X_CORNER();
		  gConfig.y_accel_corner = GET_ACCEL_Y_CORNER();
		  gConfig.z_accel_corner = GET_ACCEL_Z_CORNER();
		  gConfig.x_gyro_corner = GET_GYRO_X_CORNER();
		  gConfig.y_gyro_corner = GET_GYRO_Y_CORNER();
		  gConfig.z_gyro_corner = GET_GYRO_Z_CORNER();
		  
		  // Gyro and accelerometer default FIR filter tap lengths
		  gConfig.x_accel_taps = GET_ACCEL_X_TAPS();
		  gConfig.y_accel_taps = GET_ACCEL_Y_TAPS();
		  gConfig.z_accel_taps = GET_ACCEL_Z_TAPS();
		  gConfig.x_gyro_taps = GET_GYRO_X_TAPS();
		  gConfig.y_gyro_taps = GET_GYRO_Y_TAPS();
		  gConfig.z_gyro_taps = GET_GYRO_Z_TAPS();
		  
		  // Channel enable/disable flags
		  gConfig.x_accel_enabled = IS_ACCEL_X_ENABLED();
		  gConfig.y_accel_enabled = IS_ACCEL_Y_ENABLED();
		  gConfig.z_accel_enabled = IS_ACCEL_Z_ENABLED();
		  gConfig.x_gyro_enabled = IS_GYRO_X_ENABLED();
		  gConfig.y_gyro_enabled = IS_GYRO_Y_ENABLED();
		  gConfig.z_gyro_enabled = IS_GYRO_Z_ENABLED();

		  // Broadcast mode or listen mode
		  gConfig.broadcast_enabled = GET_TRANSMIT_MODE();
		  gConfig.broadcast_rate = GET_BROADCAST_RATE();
	  
	 }
	 
}


/*******************************************************************************
* Function Name  : WriteConfigurationToFlash
* Input          : None
* Output         : None
* Return         : None
* Description    : Writes the current IMU configuration to flash.
*******************************************************************************/
int WriteConfigurationToFlash( void )
{
	 FLASH_Status FLASHStatus;
	 uint32_t FLASHData;
	 
	 FLASH_Unlock();
	 
	 // Clear all pending flags
	 FLASH_ClearFlag(FLASH_FLAG_BSY | FLASH_FLAG_EOP | FLASH_FLAG_PGERR | FLASH_FLAG_WRPRTERR);
	 
	 // Clear FLASH memory
	 FLASHStatus = FLASH_ErasePage(OFFSET_CONF1_ADDR);
	 FLASHStatus = FLASH_ErasePage(OFFSET_CONF2_ADDR);
	 FLASHStatus = FLASH_ErasePage(OFFSET_CONF3_ADDR);
	 FLASHStatus = FLASH_ErasePage(FIR_CONF_ADDR);
	 FLASHStatus = FLASH_ErasePage(FIR_TAP_CONF_ADDR);
	 FLASHStatus = FLASH_ErasePage(USART_CONF_ADDR);
	 
	 FLASHData = ((uint32_t)gConfig.x_accel_bias) | ((uint32_t)gConfig.y_accel_bias << 16);
	 FLASHStatus = FLASH_ProgramWord(OFFSET_CONF1_ADDR, FLASHData);
	 
	 if( FLASHStatus != FLASH_COMPLETE )
	 {
		  return FLASHStatus;
	 }
	 
	 FLASHData = ((uint32_t)gConfig.z_accel_bias) | ((uint32_t)gConfig.x_gyro_bias << 16);
	 FLASHStatus = FLASH_ProgramWord(OFFSET_CONF2_ADDR, FLASHData);
	 
	 if( FLASHStatus != FLASH_COMPLETE )
	 {
		  return FLASHStatus;
	 }
	 
	 FLASHData = ((uint32_t)gConfig.y_gyro_bias) | ((uint32_t)gConfig.z_gyro_bias << 16);
	 FLASHStatus = FLASH_ProgramWord(OFFSET_CONF3_ADDR, FLASHData);
	 
	 if( FLASHStatus != FLASH_COMPLETE )
	 {
		  return FLASHStatus;
	 }
	 
	 FLASHData = ((uint32_t)gConfig.x_accel_corner) | ((uint32_t)gConfig.y_accel_corner << 4) | ((uint32_t)gConfig.z_accel_corner << 8);
	 FLASHData = FLASHData | ((uint32_t)gConfig.x_gyro_corner << 12) | ((uint32_t)gConfig.y_gyro_corner << 16) | ((uint32_t)gConfig.z_gyro_corner << 20);
	 FLASHStatus = FLASH_ProgramWord(FIR_CONF_ADDR, FLASHData);
	 
	 if( FLASHStatus != FLASH_COMPLETE )
	 {
		  return FLASHStatus;
	 }
	 
	 FLASHData = ((uint32_t)gConfig.x_accel_taps) | ((uint32_t)gConfig.y_accel_taps << 2) | ((uint32_t)gConfig.z_accel_taps << 4);
	 FLASHData = FLASHData | ((uint32_t)gConfig.x_gyro_taps << 6) | ((uint32_t)gConfig.y_gyro_taps << 8) | ((uint32_t)gConfig.z_gyro_taps << 10);
	 FLASHStatus = FLASH_ProgramWord(FIR_TAP_CONF_ADDR, FLASHData);
	 
	 if( FLASHStatus != FLASH_COMPLETE )
	 {
		  return FLASHStatus;
	 }
	 
	 FLASHData = ((uint32_t)gConfig.x_accel_enabled) | (((uint32_t)gConfig.y_accel_enabled << 1) & 0x02) | (((uint32_t)gConfig.z_accel_enabled << 2) & 0x04);
	 FLASHData = FLASHData | (((uint32_t)gConfig.x_gyro_enabled << 3) & 0x08) | (((uint32_t)gConfig.y_gyro_enabled << 4) & 0x010) | (((uint32_t)gConfig.z_gyro_enabled << 5) & 0x020);
	 FLASHData = FLASHData | (((uint32_t)gConfig.broadcast_enabled & 0x01) << 6 ) | (((uint32_t)gConfig.broadcast_rate << 8) & 0x0FF00);
	 FLASHStatus = FLASH_ProgramWord(USART_CONF_ADDR, FLASHData);
	 
	 if( FLASHStatus != FLASH_COMPLETE )
	 {
		  return FLASHStatus;
	 }
	 
	 FLASH_Lock();
	 
	 return FLASH_COMPLETE;
	 
}

/*******************************************************************************
* Function Name  : UpdateBroadcastRate
* Input          : None
* Output         : None
* Return         : None
* Description    : Sets the Timer2 period to adjust IMU broadcast frequency.
						  In Broadcast Mode, data is transmitted on every timer interrupt
*******************************************************************************/
void UpdateBroadcastRate( uint8_t new_rate )
{
	 TIM_TimeBaseInitTypeDef  TIM_TimeBaseStructure;
	 uint16_t new_period;
	 	 
	 // Update gConfig parameter
	 gConfig.broadcast_rate = new_rate;
	 
	 // Calculate new period.  The desired broadcast frequency is given by
	 // ft = (280/255)*new_rate + 20
	 // which yields rates ranging from 20 Hz to ~ 400 Hz in ~ 1.4 Hz increments.
	 // With a prescaler value of 100, a system clock of 64Mhz, and no clock
	 // division, the timer period should be:
	 // new_period = 640000/ft
	 new_period = (uint16_t)(640000.0/(1.09803921*(float)new_rate + 20.0));

	 // Update TIM1
	 TIM_TimeBaseStructure.TIM_Period = new_period;
	 TIM_TimeBaseStructure.TIM_Prescaler = 100;
	 TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	 TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;

	 TIM_TimeBaseInit(TIM2, &TIM_TimeBaseStructure);
	 	 
}

/*******************************************************************************
* Function Name  : EnableBroadcastMode
* Input          : None
* Output         : None
* Return         : None
* Description    : Turns on broadcast mode and calls UpdateBroadcastRate
*******************************************************************************/
void EnableBroadcastMode( uint8_t new_rate )
{
	 // Set global configuration variable
	 gConfig.broadcast_enabled = MODE_BROADCAST;
	 
	 // Set broadcast rate
	 UpdateBroadcastRate( new_rate );
	 
	 // Enable Timer 2
	 TIM_Cmd(TIM2, ENABLE);	 
	 
	 // Clear pending interrupt bit
	 TIM_ClearITPendingBit(TIM2, TIM_IT_Update);
	 
	 // TIM IT enable
	 TIM_ITConfig(TIM2, TIM_IT_Update, ENABLE);
}

/*******************************************************************************
* Function Name  : DisableBroadcastMode
* Input          : None
* Output         : None
* Return         : None
* Description    : Disables Broadcast Mode by turning off Timer 2
*******************************************************************************/
void DisableBroadcastMode( void )
{
	 // Set global configuration variable
	 gConfig.broadcast_enabled = MODE_LISTEN;
	 
	 // Disable Timer 2
	 TIM_Cmd( TIM2, DISABLE );	 
}

/*******************************************************************************
* Function Name  : StartGyroCalibration
* Input          : None
* Output         : None
* Return         : None
* Description    : Sets the gZeroGyroEnable flag, which triggers the FIR filtering
						  code in process_input_buffers to start summing outputs to 
						  obtain an average.  The average will be used as the new
						  zero point.
*******************************************************************************/
void StartGyroCalibration( void )
{
	 gZeroGyroSampleCount = 0;
	 gZeroGyroAverages[0] = 0;
	 gZeroGyroAverages[1] = 0;
	 gZeroGyroAverages[2] = 0;

	 // Disable broadcast timer while zeroing takes place (no packets will be sent until gyro
	 // zeroing is complete).
	 TIM_Cmd(TIM2, DISABLE);

	 // Enable gyro zeroing code.  When zeroing is complete, Timer 2 will be reactivated
	 // if the IMU is in Broadcast Mode.
	 gZeroGyroEnable = 1;
	 
}

/*******************************************************************************
* Function Name  : StopGyroCalibration
* Input          : None
* Output         : None
* Return         : None
* Description    : Finishes gyro calibration cycle. This function is called by 
						  FIR filtering software in process_input_buffers after enough
						  data has been collected to make an average.
*******************************************************************************/
void StopGyroCalibration( void )
{
	 gZeroGyroEnable = 0;
	 
	 gZeroGyroAverages[0] = (uint16_t)(gZeroGyroAverages[0]/gZeroGyroSampleCount);
	 gZeroGyroAverages[1] = (uint16_t)(gZeroGyroAverages[1]/gZeroGyroSampleCount);
	 gZeroGyroAverages[2] = (uint16_t)(gZeroGyroAverages[2]/gZeroGyroSampleCount);
	 
	 gConfig.y_gyro_bias = gZeroGyroAverages[0];
	 gConfig.x_gyro_bias = gZeroGyroAverages[1];
	 gConfig.z_gyro_bias = gZeroGyroAverages[2];
	 
	 SendGyroBiasPacket( );
	 
	 if( gConfig.broadcast_enabled )
	 {
		  TIM_Cmd(TIM2, ENABLE);
	 }	 
}


/*******************************************************************************
* Function Name  : StartSelfTest
* Input          : None
* Output         : None
* Return         : None
* Description    : Sets the gSelfTestEnable flag, which triggers the self-test
				       sequence on the IMU.
*******************************************************************************/
void StartSelfTest( void )
{
	 gSelfTestEnabled = 1;
	 gSelfTestSamplesIgnored = 0;

	 // Assert self-test pin
	 GPIO_WriteBit( GPIOA, GPIO_Pin_13, Bit_SET );
}

/*******************************************************************************
* Function Name  : StopSelfTest
* Input          : None
* Output         : None
* Return         : None
* Description    : 
*******************************************************************************/
void StopSelfTest( uint16_t result )
{
	 // Clear self-test pin
	 GPIO_WriteBit( GPIOA, GPIO_Pin_13, Bit_RESET );
	 
	 gSelfTestEnabled = 0;
	 
	 SendStatusReportPacket( result );

}
