/* ------------------------------------------------------------------------------
  File: chr6d_FIR.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Functions for performing FIR filtering  
------------------------------------------------------------------------------ */ 
#include "stm32f10x.h"
#include "chr6d_FIR.h"
#include "chr6d_startup.h"
#include "chr6d_config.h"
#include "math.h"

// Allocate space for DMA data transfer.
// Each DMA transfer returns 32 bits of data, with the first 16 from ADC1, and
// the second 16 from ADC2.  The two ADC converters make simultaneous samples.
// Allocate enough space to hold 2*SAMPLES_PER_BUFFER
// samples from each channel (for ping-pong buffer).  Since each 32-bit value 
// actually holds two ADC conversions, we need (CHANNEL_COUNT/2)*2*SAMPLES_PER_BUFFER
// 32-bit values to be stored.
__IO uint32_t ADC_DualConvertedValueTab[ CHANNEL_COUNT*SAMPLES_PER_BUFFER ];

// Set up ping-pong buffer.  When a DMA transfer is halfway complete, an interrupt will
// fire, and the data in the 'ping' buffer will be processed.  When a DMA transfer finishes
// completely, then another interrupt fires and the 'pong' buffer is processed.
__IO uint16_t *gPingBuffer = (__IO uint16_t *)ADC_DualConvertedValueTab;
__IO uint16_t *gPongBuffer = (__IO uint16_t *)(&ADC_DualConvertedValueTab[(CHANNEL_COUNT*SAMPLES_PER_BUFFER)/2]);

volatile int gPingBufferReady = 0;
volatile int gPongBufferReady = 0;

// Define arrays for storing FIR filter coefficients for 8, 16, 32, and 64 tap filters.  Filters are symmetrical, so 
// only half the coefficients need to be stored.  There are 14 corner frequencies available for each filter size,
// ranging from 10 Hz to 140 Hz in 10 Hz increments.
// The filter data stored here is signed, fixed-point data. To obtain the actual fractional value of the filter
// coefficients, divide each term by (2^15).

// 8 taps	 
short h8[14][4] = {{-290,1490,5633,9551},
						 {-982,66,5541,11759},
						 {-1208,-280,5552,12320},
						 {-1086,-1291,5075,13686},
						 {-17,-2235,2677,15959},
						 {496,-2426,1241,17072},
						 {875,-2520,407,17622},
						 {1148,-2092,-548,17876},
						 {557,-206,-3001,19034},
						 {-352,1585,-5013,20164},
						 {-491,1828,-5268,20314},
						 {-717,2134,-5567,20533},
						 {-1104,2523,-5934,20899},
						 {-1823,3034,-6428,21600}};
						 
// 16 taps						  
short h16[14][8] = {{-367,-546,-465,238,1681,3629,5522,6692},
						   {30,-476,-862,-707,649,3247,6242,8261},
							{328,153,-613,-1441,-873,2112,6640,10078},
							{100,559,143,-1181,-1828,732,6442,11416},
							{-237,290,854,-176,-2213,-964,5832,12998},
							{-222,-327,783,817,-1785,-2321,4964,14475},
						   {133,-518,-127,1403,-460,-3365,3409,15909},
							{266,-69,-785,1088,803,-3579,1864,16797},
							{-13,460,-694,-58,1913,-3041,-71,17887},
							{-280,423,-75,-1008,2240,-2098,-1695,18876},
							{-97,-182,708,-1360,1567,-440,-3479,19667},
							{-22,97,-290,698,-1476,2939,-6140,20578},
							{-52,170,-420,884,-1694,3148,-6294,20643},
							{-138,305,-619,1131,-1957,3386,-6467,20744}};
							
// 32 taps
short h32[14][16] = {{-14,-79,-145,-245,-343,-407,-390,-247,59,545,1193,1952,2740,3458,4005,4302},
						  {49,119,187,227,160,-34,-354,-694,-905,-787,-199,899,2375,3963,5304,6074},
						  {-62,-119,-134,-35,189,436,508,226,-410,-1113,-1371,-683,1127,3711,6263,7850},
						  {59,94,36,-150,-335,-275,146,669,739,-25,-1275,-1883,-648,2669,6834,9730},
						  {-52,-42,89,247,171,-222,-530,-194,682,1063,15,-1797,-2060,1091,6721,11203},
						  {41,-5,-154,-178,124,392,31,-662,-475,810,1306,-567,-2729,-759,6256,12953},
						  {-23,65,168,-6,-292,-44,514,175,-840,-438,1340,975,-2260,-2355,5209,14195},
						  {1,-114,-107,177,146,-366,-163,678,85,-1161,200,1936,-981,-3540,3916,15678},
						  {26,133,-5,-205,161,261,-476,-104,880,-476,-1121,1678,688,-3952,2251,16645},
						  {-54,-124,112,75,-300,212,285,-693,324,780,-1462,421,2113,-3677,477,17896},
						  {109,416,547,83,-491,-164,650,321,-915,-597,1332,1115,-2141,-2413,4883,13650},
						  {-105,-21,129,-227,201,24,-395,705,-670,99,928,-1974,2314,-1068,-3095,19540},
						  {122,-47,-26,158,-305,389,-321,43,440,-1024,1505,-1608,1013,695,-4559,19911},
						  {-95,60,-39,-27,145,-311,507,-696,825,-826,622,-120,-814,2468,-5877,20563}};

// 64 taps
 short h64[14][32] = {{2,7,13,24,37,54,71,85,94,93,77,43,-11,-85,-173,-268,-358,-429,-463,-443,-356,-189,61,392,793,1244,1720,2188,2615,2969,3222,3354},
							 {5,11,19,25,27,18,-3,-39,-82,-121,-142,-128,-70,32,160,285,365,360,241,8,-307,-635,-882,-943,-734,-206,627,1691,2849,3935,4778,5238},
							 {4,6,4,-8,-29,-54,-70,-62,-18,53,127,162,123,2,-166,-306,-331,-188,106,442,655,589,181,-471,-1114,-1402,-1028,141,1976,4089,5945,7028},
							 {1,-6,-21,-41,-50,-32,17,73,95,47,-60,-163,-171,-42,170,318,257,-37,-399,-555,-307,278,835,896,226,-910,-1770,-1499,350,3437,6685,8761},
							 {-5,-22,-39,-38,-4,47,67,18,-75,-121,-45,114,202,87,-170,-321,-153,249,497,252,-365,-765,-409,554,1221,693,-921,-2218,-1418,2099,6943,10429},
							 {-10,-23,-20,16,58,49,-24,-79,-24,102,131,-25,-198,-126,169,310,27,-391,-348,243,657,201,-720,-848,305,1441,755,-1547,-2621,342,6653,11928},
							 {-7,-5,23,54,32,-39,-56,34,100,-2,-143,-62,172,159,-167,-285,107,424,29,-553,-259,636,600,-623,-1074,442,1741,61,-2828,-1461,5934,13399},
							 {-1,18,48,31,-35,-41,48,62,-67,-93,92,134,-125,-186,168,252,-226,-337,304,445,-410,-588,561,787,-787,-1089,1168,1629,-1960,-2973,4790,14763},
							 {8,38,42,-18,-40,40,44,-74,-36,121,5,-171,62,210,-170,-214,316,155,-485,-1,646,-274,-749,697,722,-1303,-452,2196,-343,-3901,3302,16015},
							 {18,47,12,-41,17,46,-61,-19,104,-56,-103,161,12,-230,169,172,-366,70,430,-445,-204,761,-359,-729,1110,69,-1671,1404,1409,-4057,1572,17142},
							 {28,43,-21,-17,50,-35,-32,89,-59,-57,151,-100,-88,242,-168,-125,373,-276,-165,563,-453,-204,853,-759,-239,1362,-1382,-264,2625,-3405,-278,18135},
							 {30,11,-50,16,-1,-61,65,-36,-58,125,-123,5,152,-247,165,74,-337,411,-183,-266,647,-640,120,677,-1226,997,160,-1781,2848,-2084,-2121,19093},
							 {17,-35,-37,34,-48,27,17,-73,109,-98,26,88,-196,236,-164,-19,253,-432,446,-235,-164,610,-894,819,-295,-594,1565,-2188,1960,-329,-3805,19781},
							 {13,-50,0,6,-35,60,-76,73,-44,-13,88,-163,214,-219,158,-31,-148,340,-497,561,-486,250,137,-619,1102,-1462,1554,-1228,309,1502,-5220,20305}};

// Variables used for automatic gyro bias calibration
uint16_t gZeroGyroEnable;
uint32_t gZeroGyroAverages[3];
uint16_t gZeroGyroSampleCount;
						  
// Variables used for automatic self-test feature
uint16_t gSelfTestEnabled;
int16_t gSelfTestResults[6];
uint16_t gSelfTestSamplesIgnored;
						  
/*******************************************************************************
* Function Name  : process_input_buffers
* Input          : short* data_output
* Output         : short FIR_input[][MAX_INPUT_SIZE], short* signed_decimated
* Return         : 1 if new data was available, 0 otherwise
* Description    : 
						  
If new ADC data is available (ie. the ping buffer or the pong buffer is filled),
then this function processes the new data.

First, the ADC data in the ping or pong buffer is decimated.  Then, gyro and
accelerometer biases are subtracted to create signed versions of the ADC data
that are appropriate for input to the FIR filter.  Then, an FIR filter is run
on each data channel.
						  
The buffer FIR_input[][MAX_INPUT_SIZE] is where the sensor data used as input to the FIR
filter is stored.  This buffer is actually not filled by the calling function,
but by process_input_buffers (this function).  The buffer is stored external
to this function because in some cases it can be useful to access the unfiltered
data.

The buffer signed_decimated* is also filled by this function, not by the calling
function.  This buffer stores the signed, decimated ADC data, before it is run
through the FIR filter.  Again, this buffer is stored external to the function
because it may be useful to access the data.

Finally, the buffer data_output is also filled by this function, and is stored externally
(outside the the process_input_buffers function).  This is the actual output of the
FIR filter.  If the FIR filter is disabled, the data_output buffer data is equivalent
to the signed_decimated buffer data.

This function also handles automatic bias calculation for the rate gyros.

*******************************************************************************/
int process_input_buffers( short FIR_input[][MAX_INPUT_SIZE], short* signed_decimated, short* data_output )
{
	 int FIR_output;
	 int i,j;
	 int taps;
	 
	 short* h;
	 
	 __IO uint16_t* input_buffer;
	 
	 uint16_t decimated_output[CHANNEL_COUNT];
	 
	 // ----------------------------------------------------------------------------------
	 // Determine whether gPingBuffer or gPongBuffer are ready.  If not, return 0
	 // ----------------------------------------------------------------------------------
	 if( !gPingBufferReady && !gPongBufferReady ) 
	 {
		  return 0;
	 }
	 else
	 {
		  // Determine which buffer is ready.
		  if( gPingBufferReady )
		  {
				gPingBufferReady = 0;
				input_buffer = gPingBuffer;
		  }
		  else
		  {
				gPongBufferReady = 0;
				input_buffer = gPongBuffer;
		  }
		  
		  // Decimate ADC input (reduces quantization noise and increases resolution)
		  decimate( decimated_output, input_buffer );
		  
		  // Convert averages to 2's complement signed numbers
		  signed_decimated[0] = (short)(decimated_output[0]-gConfig.x_accel_bias);
		  signed_decimated[1] = (short)(decimated_output[1]-gConfig.y_accel_bias);
		  signed_decimated[2] = (short)(decimated_output[2]-gConfig.z_accel_bias);
		  signed_decimated[3] = (short)(decimated_output[3]-gConfig.y_gyro_bias);
		  signed_decimated[4] = (short)(decimated_output[4]-gConfig.x_gyro_bias);
		  signed_decimated[5] = (short)(decimated_output[5]-gConfig.z_gyro_bias);
		  
		  
		  // If the "zero gyro" command is enabled, then add the newest measurement
		  // to the avg. buffer
		  if( gZeroGyroEnable )
		  {
				gZeroGyroSampleCount++;
				
				gZeroGyroAverages[0] += decimated_output[3];
				gZeroGyroAverages[1] += decimated_output[4];
				gZeroGyroAverages[2] += decimated_output[5];
				
				if( gZeroGyroSampleCount == GYRO_ZERO_SAMPLE_SIZE )
				{
					 StopGyroCalibration( );
				}
		  }
		  
		  // Self-test command
		  if( gSelfTestEnabled )
		  {
				// If self test was just enabled, sample
				// current outputs of sensors for comparison
				if( gSelfTestSamplesIgnored == 0 )
				{
					 gSelfTestResults[0] = signed_decimated[0];
					 gSelfTestResults[1] = signed_decimated[1];
					 gSelfTestResults[2] = signed_decimated[2];
					 gSelfTestResults[3] = signed_decimated[3];
					 gSelfTestResults[4] = signed_decimated[4];
					 gSelfTestResults[5] = signed_decimated[5];
				}
				// ignore SELF_TEST_IGNORE_SAMPLES beforetesting output for change. 
				// This allows the sensors time to react to the self-test signal.
				else if( gSelfTestSamplesIgnored == SELF_TEST_IGNORE_SAMPLES )
				{
					 gSelfTestResults[0] -= signed_decimated[0];
					 gSelfTestResults[1] -= signed_decimated[1];
					 gSelfTestResults[2] -= signed_decimated[2];
					 gSelfTestResults[3] -= signed_decimated[3];
					 gSelfTestResults[4] -= signed_decimated[4];
					 gSelfTestResults[5] -= signed_decimated[5];
					 
					 uint16_t self_test_result = 0;
					 
					 // Check that the sensor outputs responded to the self-test signal
					 if( abs(gSelfTestResults[0]) < SELF_TEST_ACCEL_THRESHOLD )
					 {
						  self_test_result = 0x01;
					 }
					 if( abs(gSelfTestResults[1]) < SELF_TEST_ACCEL_THRESHOLD )
					 {
						  self_test_result = self_test_result | 0x02;
					 }
					 if( abs(gSelfTestResults[2]) < SELF_TEST_ACCEL_THRESHOLD )
					 {
						  self_test_result = self_test_result | 0x04;
					 }
					 if( abs(gSelfTestResults[3]) < SELF_TEST_GYRO_THRESHOLD )
					 {
						  self_test_result = self_test_result | 0x08;
					 }
					 if( abs(gSelfTestResults[4]) < SELF_TEST_GYRO_THRESHOLD )
					 {
						  self_test_result = self_test_result | 0x10;
					 }
					 if( abs(gSelfTestResults[5]) < SELF_TEST_GYRO_THRESHOLD )
					 {
						  self_test_result = self_test_result | 0x20;
					 }
					 
					 // Stop self test and send result packet
					 StopSelfTest( self_test_result );
				}
				
				gSelfTestSamplesIgnored++;
		  }

		  // Add new decimated output to the end of each FIR input buffer.  
		  // Shift all existing entries down one.
		  for( i = 0; i < CHANNEL_COUNT; i++ )
		  {
				int input_size;
				
				// Determine the input size based on the number of taps used
				// in each filter.  The filter tap count is stored where
				// 0 = 8 taps, 1 = 16 taps, 2 = 32 taps, 3 = 64 taps.
				// To obtain the tap count using the stored values, we use
				// input_size = (8 << tap_setting).
				switch( i )
				{
					 case 0:
						  input_size = (8 << gConfig.x_accel_taps);
						  break;
					 case 1:
						  input_size = (8 << gConfig.y_accel_taps);
						  break;
					 case 2:
						  input_size = (8 << gConfig.z_accel_taps);
						  break;
					 case 3:
						  input_size = (8 << gConfig.y_gyro_taps);
						  break;
					 case 4:
						  input_size = (8 << gConfig.x_gyro_taps);
						  break;
					 case 5:
						  input_size = (8 << gConfig.z_gyro_taps);
						  break;
					 
				}
				
				for( j = 0; j < (input_size-1); j++ )
				{
					 FIR_input[i][j] = FIR_input[i][j+1];
				}
				
				FIR_input[i][input_size-1] = signed_decimated[i];
		  }
				
		  // Using new input data, run FIR filter on each channel.  Convert result
		  // back into a 16 bit value.
		  
		  // X accelerometer
		  if( gConfig.x_accel_corner != FIR_DISABLED )
		  {
				taps = 8 << gConfig.x_accel_taps;
				
				// Select the correct filter coefficients
				switch( gConfig.x_accel_taps )
				{
					 case FIR_TAPS_8:
						  h = h8[gConfig.x_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_16:
						  h = h16[gConfig.x_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_32:
						  h = h32[gConfig.x_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_64:
						  h = h64[gConfig.x_accel_corner-2];
					 break;
				}
				
				// Execute filter
				FIR_output = single_point_FIR( FIR_input[0], h, taps );
				data_output[0] = (short)( ((FIR_output*2) >> 16 ) & 0x0FFFF );
		  }
		  else
		  {
				data_output[0] = signed_decimated[0];
		  }
		  
		  // Y accelerometer
		  if( gConfig.y_accel_corner != FIR_DISABLED )
		  {
				taps = 8 << gConfig.y_accel_taps;
				
				// Select the correct filter coefficients
				switch( gConfig.y_accel_taps )
				{
					 case FIR_TAPS_8:
						  h = h8[gConfig.y_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_16:
						  h = h16[gConfig.y_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_32:
						  h = h32[gConfig.y_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_64:
						  h = h64[gConfig.y_accel_corner-2];
					 break;
				}
				
				FIR_output = single_point_FIR( FIR_input[1], h, taps );
				data_output[1] = (short)( ((FIR_output*2) >> 16 ) & 0x0FFFF );
		  }
		  else
		  {
				data_output[1] = signed_decimated[1];
		  }
		
		  // Z accelerometer
		  if( gConfig.z_accel_corner != FIR_DISABLED )
		  {
				taps = 8 << gConfig.z_accel_taps;
				
				// Select the correct filter coefficients
				switch( gConfig.z_accel_taps )
				{
					 case FIR_TAPS_8:
						  h = h8[gConfig.z_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_16:
						  h = h16[gConfig.z_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_32:
						  h = h32[gConfig.z_accel_corner-2];
					 break;
					 
					 case FIR_TAPS_64:
						  h = h64[gConfig.z_accel_corner-2];
					 break;
				}
				
				FIR_output = single_point_FIR( FIR_input[2], h, taps );
				data_output[2] = (short)( ((FIR_output*2) >> 16 ) & 0x0FFFF );
		  }
		  else
		  {
				data_output[2] = signed_decimated[2];
		  }
		  
		  // Y rate gyro
		  if( gConfig.y_gyro_corner != FIR_DISABLED )
		  {
				taps = 8 << gConfig.y_gyro_taps;
				
				// Select the correct filter coefficients
				switch( gConfig.y_gyro_taps )
				{
					 case FIR_TAPS_8:
						  h = h8[gConfig.y_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_16:
						  h = h16[gConfig.y_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_32:
						  h = h32[gConfig.y_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_64:
						  h = h64[gConfig.y_gyro_corner-2];
					 break;
				}
				
				FIR_output = single_point_FIR( FIR_input[3], h, taps );
				data_output[3] = (short)( ((FIR_output*2) >> 16 ) & 0x0FFFF );
		  }
		  else
		  {	
				data_output[3] = signed_decimated[3];
		  }
		  
		  // X rate gyro
		  if( gConfig.x_gyro_corner != FIR_DISABLED )
		  {
				taps = 8 << gConfig.x_gyro_taps;
				
				// Select the correct filter coefficients
				switch( gConfig.x_gyro_taps )
				{
					 case FIR_TAPS_8:
						  h = h8[gConfig.x_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_16:
						  h = h16[gConfig.x_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_32:
						  h = h32[gConfig.x_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_64:
						  h = h64[gConfig.x_gyro_corner-2];
					 break;
				}
				
				FIR_output = single_point_FIR( FIR_input[4], h, taps );
				data_output[4] = (short)( ((FIR_output*2) >> 16 ) & 0x0FFFF );
		  }
		  else
		  {
				data_output[4] = signed_decimated[4];
		  }
		  
		  // Z rate gyro
		  if( gConfig.z_gyro_corner != FIR_DISABLED )
		  {
				taps = 8 << gConfig.z_gyro_taps;
				
				// Select the correct filter coefficients
				switch( gConfig.z_gyro_taps )
				{
					 case FIR_TAPS_8:
						  h = h8[gConfig.z_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_16:
						  h = h16[gConfig.z_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_32:
						  h = h32[gConfig.z_gyro_corner-2];
					 break;
					 
					 case FIR_TAPS_64:
						  h = h64[gConfig.z_gyro_corner-2];
					 break;
				}
				
				FIR_output = single_point_FIR( FIR_input[5], h, taps );
				data_output[5] = (short)( ((FIR_output*2) >> 16 ) & 0x0FFFF );
		  }
		  else
		  {
				data_output[5] = signed_decimated[5];
		  }
	 }
	 
	 return 1;
}

/*******************************************************************************
* Function Name  : decimate
* Input          : __IO uint16_t* input_buffer
* Output         : uint16_t* decimated_output
* Return         : None
* Description    : Decimates the input data, and converts raw sensor data from
						 12-bit to 16-bit.
*******************************************************************************/
void decimate( uint16_t* decimated_output, __IO uint16_t* input_buffer )
{
	 int i, j;
	 uint32_t sum_data[CHANNEL_COUNT];
	 
	 // Pre-initialize sum data
	 for( i = 0; i < CHANNEL_COUNT; i++ )
	 {
		  sum_data[i] = 0;
	 }
	 
	 // Compute sums of all channels.
	 for( i = 0; i < SAMPLES_PER_BUFFER; i++ )
	 {
		  sum_data[0] += (uint32_t)input_buffer[CHANNEL_COUNT*i];
		  sum_data[1] += (uint32_t)input_buffer[CHANNEL_COUNT*i + 1];
		  sum_data[2] += (uint32_t)input_buffer[CHANNEL_COUNT*i + 2];
		  sum_data[3] += (uint32_t)input_buffer[CHANNEL_COUNT*i + 3];
		  sum_data[4] += (uint32_t)input_buffer[CHANNEL_COUNT*i + 4];
		  sum_data[5] += (uint32_t)input_buffer[CHANNEL_COUNT*i + 5];				
	 }
 
	 // Decimate to obtain 16 bits resolution
	 decimated_output[0] = (uint16_t)(sum_data[0] >> 5);
	 decimated_output[1] = (uint16_t)(sum_data[1] >> 5);
	 decimated_output[2] = (uint16_t)(sum_data[2] >> 5);
	 decimated_output[3] = (uint16_t)(sum_data[3] >> 5);
	 decimated_output[4] = (uint16_t)(sum_data[4] >> 5);
	 decimated_output[5] = (uint16_t)(sum_data[5] >> 5);
}

/*******************************************************************************
* Function Name  : single_point_FIR
* Input          : short* x, short* h, int taps
* Output         : None
* Return         : The 32-bit signed filtered data point
* Description    : 

Performs a single point FIR of the data pointed to by x, using the filter coefficients
pointed to by h.  'taps' is the number of taps used in the filter.  This function
call assumes that the filter coefficients are symmetric, so that only half the
coefficients are stored in the memory location pointed to by h.  The size of the
buffer pointed to by x must be equal to the number of taps in the filter.

x and h are 16-bit signed, fixed-point data types.  To obtain the 16-bit version
of the filter output, the output should be multiplied by two, and then right-shifted
by 16.

*******************************************************************************/
int single_point_FIR( short* x, short* h, int taps )
{
	 int j, sum, taps_div_2;
	 int h_index, x_index;
	 
	 taps_div_2 = taps >> 1;
	 
	 sum = 0;

	 // Break the convolution sum into two halves, since we only store half of the symmetric 
	 // filter impulse response.
	 for( j=0; j < taps_div_2; j++ )
	 {
		  sum += h[j]*x[j];
		  sum += h[taps_div_2 - j - 1]*x[taps_div_2 + j];
	 }	  

	 return sum;
}