using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO.Ports;
using ZedGraph;
using System.Threading;

namespace IMU
{
    public partial class IMU : Form
    {
        int ACCEL_MAX = (int)Math.Pow(2,15);
        int ACCEL_MIN = -1 * (int)Math.Pow(2, 15);
        int GYRO_MAX = (int)Math.Pow(2, 15);
        int GYRO_MIN = -1 * (int)Math.Pow(2, 15);
        int NUM_POINTS = 200;

        const int RX_BUF_SIZE = 500;

        delegate void AppendTextCallback(string text, Color text_color);
        delegate void displayGyroBiasesCallback(uint x_gyro, uint y_gyro, uint z_gyro);
        delegate void displayAccelBiasesCallback(uint x_accel, uint y_accel, uint z_accel);
        delegate void displayFilterTapsCallback(uint x_accel, uint y_accel, uint z_accel, uint x_gyro, uint y_gyro, uint z_gyro);
        delegate void displayFilterCornersCallback(uint x_accel, uint y_accel, uint z_accel, uint x_gyro, uint y_gyro, uint z_gyro);
        delegate void displayActiveChannelsCallback(int x_accel_active, int y_accel_active, int z_accel_active, int x_gyro_active, int y_gyro_active, int z_gyro_active);
        delegate void displayTransmitModeCallback(int broadcast_enabled, int broadcast_rate);
        delegate void displayAngleEstimateCallback( double angle_estimate );

        private byte[] RXbuffer = new byte[RX_BUF_SIZE];
        private int RXbufPtr;

        GraphPane x_accel_pane, y_accel_pane, z_accel_pane;
        GraphPane x_gyro_pane, y_gyro_pane, z_gyro_pane;
        RollingPointPairList x_accel_list, y_accel_list, z_accel_list;
        RollingPointPairList x_gyro_list, y_gyro_list, z_gyro_list;
        LineItem x_accel_line, y_accel_line, z_accel_line;
        LineItem x_gyro_line, y_gyro_line, z_gyro_line;

        int num_changes;
        //Variables to store the IMU settings
        bool x_accel, y_accel, z_accel;
        bool x_gyro, y_gyro, z_gyro;

        int x_accel_taps, y_accel_taps, z_accel_taps;
        int x_gyro_taps, y_gyro_taps, z_gyro_taps;

        int x_accel_freq, y_accel_freq, z_accel_freq;
        int x_gyro_freq, y_gyro_freq, z_gyro_freq;

        SerialPort serialPort;

        double angle_estimate = 0.0;

        double test = 0.0;
        double time = 0.0;

        double[] recent_data;
        public IMU()
        {
            InitializeComponent();
            InitializeFilter();
            InitializeGraphs();
            InitializeSerialPort();
            SaveSettings();

            num_changes = 0;

            recent_data = new double[6];
            for (int i = 0; i < 6; i++)
                recent_data[i] = 0;

            
            timer1.Interval = 50;
           // timer1.Start();
            //timer2.Interval = 20;
            //timer2.Start();


            // Add even handler
            serialPort.DataReceived += new SerialDataReceivedEventHandler(serialPort_DataReceived);

            RXbufPtr = 0;
        }
        private void InitializeGraphs()
        {
            x_accel_pane = zedGraphControlXAccel.GraphPane;
            x_accel_pane.Title.Text = "X Accel";
            x_accel_pane.XAxis.Title.Text = "Time";
            x_accel_pane.YAxis.Title.Text = "Unscaled Sensor Output";
            //x_accel_pane.YAxis.Scale.Max = ACCEL_MAX;
            //x_accel_pane.YAxis.Scale.Min = ACCEL_MIN;

            y_accel_pane = zedGraphControlYAccel.GraphPane;
            y_accel_pane.Title.Text = "Y Accel";
            y_accel_pane.XAxis.Title.Text = "Time";
            y_accel_pane.YAxis.Title.Text = "Unscaled Sensor Output";
            //y_accel_pane.YAxis.Scale.Max = ACCEL_MAX;
            //y_accel_pane.YAxis.Scale.Min = ACCEL_MIN;

            z_accel_pane = zedGraphControlZAccel.GraphPane;
            z_accel_pane.Title.Text = "Z Accel";
            z_accel_pane.XAxis.Title.Text = "Time";
            z_accel_pane.YAxis.Title.Text = "Unscaled Sensor Output";
            //z_accel_pane.YAxis.Scale.Max = ACCEL_MAX;
            //z_accel_pane.YAxis.Scale.Min = ACCEL_MIN;

            x_gyro_pane = zedGraphControlXGyro.GraphPane;
            x_gyro_pane.Title.Text = "X Gyro";
            x_gyro_pane.XAxis.Title.Text = "Time";
            x_gyro_pane.YAxis.Title.Text = "Unscaled Sensor Output";
            //x_gyro_pane.YAxis.Scale.Max = GYRO_MAX;
            //x_gyro_pane.YAxis.Scale.Min = GYRO_MIN;

            y_gyro_pane = zedGraphControlYGyro.GraphPane;
            y_gyro_pane.Title.Text = "Y Gyro";
            y_gyro_pane.XAxis.Title.Text = "Time";
            y_gyro_pane.YAxis.Title.Text = "Unscaled Sensor Output";
            //y_gyro_pane.YAxis.Scale.Max = GYRO_MAX;
            //y_gyro_pane.YAxis.Scale.Min = GYRO_MIN;

            z_gyro_pane = zedGraphControlZGyro.GraphPane;
            z_gyro_pane.Title.Text = "Z Gyro";
            z_gyro_pane.XAxis.Title.Text = "Time";
            z_gyro_pane.YAxis.Title.Text = "Unscaled Sensor Output";
            //z_gyro_pane.YAxis.Scale.Max = GYRO_MAX;
            //z_gyro_pane.YAxis.Scale.Min = GYRO_MIN;

            x_accel_list = new RollingPointPairList(NUM_POINTS);
            y_accel_list = new RollingPointPairList(NUM_POINTS);
            z_accel_list = new RollingPointPairList(NUM_POINTS);
            x_gyro_list = new RollingPointPairList(NUM_POINTS);
            y_gyro_list = new RollingPointPairList(NUM_POINTS);
            z_gyro_list = new RollingPointPairList(NUM_POINTS);

            x_accel_line = x_accel_pane.AddCurve("", x_accel_list, Color.Blue, SymbolType.None);
            y_accel_line = y_accel_pane.AddCurve("", y_accel_list, Color.Blue, SymbolType.None);
            z_accel_line = z_accel_pane.AddCurve("", z_accel_list, Color.Blue, SymbolType.None);
            x_gyro_line = x_gyro_pane.AddCurve("", x_gyro_list, Color.Blue, SymbolType.None);
            y_gyro_line = y_gyro_pane.AddCurve("", y_gyro_list, Color.Blue, SymbolType.None);
            z_gyro_line = z_gyro_pane.AddCurve("", z_gyro_list, Color.Blue, SymbolType.None);
        }
        private void InitializeSerialPort()
        {
            serialPort = new SerialPort();

            foreach (string s in SerialPort.GetPortNames())
            {
                toolStripComboBoxPort.Items.Add(s);
            }
            if (toolStripComboBoxPort.Items.Count == 0)
            {
                toolStripComboBoxPort.Items.Add("No Ports Avaliable");
                serialConnectButton.Enabled = false;
            }

            toolStripComboBoxPort.SelectedIndex = 0;
            toolStripComboBoxBaud.Items.Add(115200);  // 115200, 38400
            toolStripComboBoxBaud.Items.Add(38400);
            toolStripComboBoxBaud.Items.Add(9600);
            toolStripComboBoxBaud.SelectedIndex = 0;
        }
        private void InitializeFilter()
        {
            comboBoxXGyro.Items.Add(8);
            comboBoxXGyro.Items.Add(16);
            comboBoxXGyro.Items.Add(32);
            comboBoxXGyro.Items.Add(64);
            comboBoxXGyro.SelectedIndex = 0;

            comboBoxYGyro.Items.Add(8);
            comboBoxYGyro.Items.Add(16);
            comboBoxYGyro.Items.Add(32);
            comboBoxYGyro.Items.Add(64);
            comboBoxYGyro.SelectedIndex = 0;

            comboBoxZGyro.Items.Add(8);
            comboBoxZGyro.Items.Add(16);
            comboBoxZGyro.Items.Add(32);
            comboBoxZGyro.Items.Add(64);
            comboBoxZGyro.SelectedIndex = 0;

            comboBoxXAccel.Items.Add(8);
            comboBoxXAccel.Items.Add(16);
            comboBoxXAccel.Items.Add(32);
            comboBoxXAccel.Items.Add(64);
            comboBoxXAccel.SelectedIndex = 0;

            comboBoxYAccel.Items.Add(8);
            comboBoxYAccel.Items.Add(16);
            comboBoxYAccel.Items.Add(32);
            comboBoxYAccel.Items.Add(64);
            comboBoxYAccel.SelectedIndex = 0;

            comboBoxZAccel.Items.Add(8);
            comboBoxZAccel.Items.Add(16);
            comboBoxZAccel.Items.Add(32);
            comboBoxZAccel.Items.Add(64);
            comboBoxZAccel.SelectedIndex = 0;

        }
        
        void serialPort_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            int continue_parsing = 1;

            int bytes_to_read = serialPort.BytesToRead;

            if ((RXbufPtr + bytes_to_read) >= RX_BUF_SIZE)
            {
                RXbufPtr = 0;
            }

            if (bytes_to_read >= RX_BUF_SIZE)
            {
                bytes_to_read = RX_BUF_SIZE - 1;
            }

            // Get serial data
            serialPort.Read(RXbuffer, RXbufPtr, bytes_to_read);

            RXbufPtr += bytes_to_read;

            bool found_packet;
            int packet_start_index;
            int packet_index;
            
            // If there are enough bytes in the buffer to construct a full packet, then check data.
            // There are RXbufPtr bytes in the buffer at any given time
            while (RXbufPtr >= 7 && (continue_parsing == 1))
            {
                // Search for the packet start sequence
                found_packet = false;
                packet_start_index = 0;
                for (packet_index = 0; packet_index < (RXbufPtr - 2); packet_index++)
                {
                    if (RXbuffer[packet_index] == 's' && RXbuffer[packet_index + 1] == 'n' && RXbuffer[packet_index + 2] == 'p')
                    {
                        found_packet = true;
                        packet_start_index = packet_index;

                        break;
                    }
                }

                // If start sequence found, try to recover all the data in the packet
                if (found_packet && ((RXbufPtr - packet_start_index) >= 7))
                {
                    byte packet_type = RXbuffer[packet_start_index + 3];
                    byte data_size = RXbuffer[packet_start_index + 4];
                    byte[] data = new byte[20];

                    // If a full packet has been received, then the full packet size should be
                    // 3 + 1 + 1 + [data_size] + 2
                    // that is, 3 bytes for the start sequence, 1 byte for type, 1 byte for data length, data_size bytes
                    // for packet data, and 2 bytes for the checksum.
                    // If enough data has been received, go ahead and recover the packet.  If not, wait until the
                    // rest of the data arrives
                    int buffer_length = (RXbufPtr - packet_start_index);
                    int packet_length = (7 + data_size);
                    if (buffer_length >= packet_length)
                    {
                        // A full packet has been received.  Retrieve the data.
                        for (int i = 0; i < data_size; i++)
                        {
                            data[i] = RXbuffer[packet_start_index + 5 + i];
                        }

                        handle_packet(packet_type, (int)data_size, data);

                        // Copy all received bytes that weren't part of this packet into the beginning of the
                        // buffer.  Then, reset RXbufPtr.
                        for (int index = 0; index < (buffer_length - packet_length); index++)
                        {
                            RXbuffer[index] = RXbuffer[(packet_start_index + packet_length) + index];
                        }

                        RXbufPtr = (buffer_length - packet_length);

                        //                        AppendStatusText("\r\nReceived " + packet_length.ToString() + " bytes.", Color.Green);
                    }
                    else
                    {
                        continue_parsing = 0;
                    }
                }
                else
                {
                    continue_parsing = 0;
                }
            }
            //            AppendStatusText("\r\nReceived " + bytes.ToString() + " bytes.", Color.Green);
        }

        private void AppendStatusText(string text, Color text_color)
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (this.StatusBox.InvokeRequired)
            {
                AppendTextCallback d = new AppendTextCallback(AppendStatusText);
                this.Invoke(d, new object[] { text, text_color });
            }
            else
            {
                this.StatusBox.SelectionColor = text_color;
                this.StatusBox.AppendText(text);
                this.StatusBox.ScrollToCaret();
            }
        }

        private void displayGyroBiases( uint x_gyro, uint y_gyro, uint z_gyro )
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (this.biasXGyro.InvokeRequired)
            {
                displayGyroBiasesCallback d = new displayGyroBiasesCallback(displayGyroBiases);
                this.Invoke(d, new object[] { x_gyro, y_gyro, z_gyro });
            }
            else
            {
                this.biasXGyro.Text = x_gyro.ToString();
                this.biasYGyro.Text = y_gyro.ToString();
                this.biasZGyro.Text = z_gyro.ToString();

            }
        }

        private void displayAccelBiases(uint x_accel, uint y_accel, uint z_accel)
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (this.biasXAccel.InvokeRequired)
            {
                displayAccelBiasesCallback d = new displayAccelBiasesCallback(displayAccelBiases);
                this.Invoke(d, new object[] { x_accel, y_accel, z_accel });
            }
            else
            {
                this.biasXAccel.Text = x_accel.ToString();
                this.biasYAccel.Text = y_accel.ToString();
                this.biasZAccel.Text = z_accel.ToString();

            }
        }


        private void displayFilterTaps(uint x_accel, uint y_accel, uint z_accel, uint x_gyro, uint y_gyro, uint z_gyro)
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (this.comboBoxXAccel.InvokeRequired)
            {
                displayFilterTapsCallback d = new displayFilterTapsCallback(displayFilterTaps);
                this.Invoke(d, new object[] { x_accel, y_accel, z_accel, x_gyro, y_gyro, z_gyro });
            }
            else
            {
                this.comboBoxXAccel.SelectedIndex = (int)x_accel;
                this.comboBoxYAccel.SelectedIndex = (int)y_accel;
                this.comboBoxZAccel.SelectedIndex = (int)z_accel;
                this.comboBoxXGyro.SelectedIndex = (int)x_gyro;
                this.comboBoxYGyro.SelectedIndex = (int)y_gyro;
                this.comboBoxZGyro.SelectedIndex = (int)z_gyro;
            }
        }

        private void displayFilterCorners(uint x_accel, uint y_accel, uint z_accel, uint x_gyro, uint y_gyro, uint z_gyro)
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (this.cornerXAccel.InvokeRequired)
            {
                displayFilterCornersCallback d = new displayFilterCornersCallback(displayFilterCorners);
                this.Invoke(d, new object[] { x_accel, y_accel, z_accel, x_gyro, y_gyro, z_gyro });
            }
            else
            {
                if( x_accel >= 2 )
                    this.cornerXAccel.Value = (int)(x_accel-1)*10;
                else
                    this.cornerXAccel.Value = 0;
                if( y_accel >= 2 )
                    this.cornerYAccel.Value = (int)(y_accel - 1)*10;
                else
                    this.cornerYAccel.Value = 0;
                if (z_accel >= 2)
                    this.cornerZAccel.Value = (int)(z_accel - 1) * 10;
                else
                    this.cornerZAccel.Value = 0;
                if (x_gyro >= 2)
                    this.cornerXGyro.Value = (int)(x_gyro - 1) * 10;
                else
                    this.cornerXGyro.Value = 0;
                if( y_gyro >= 2 )
                    this.cornerYGyro.Value = (int)(y_gyro - 1) * 10;
                else
                    this.cornerYGyro.Value = 0;
                if( z_gyro >= 2 )
                    this.cornerZGyro.Value = (int)(z_gyro - 1) * 10;
                else
                    this.cornerZGyro.Value = 0;

            }
        }

        private void displayActiveChannels(int x_accel_active, int y_accel_active, int z_accel_active, int x_gyro_active, int y_gyro_active, int z_gyro_active)
        {
            if (this.checkBoxXAccel.InvokeRequired)
            {
                displayActiveChannelsCallback d = new displayActiveChannelsCallback(displayActiveChannels);
                this.Invoke(d, new object[] { x_accel_active, y_accel_active, z_accel_active, x_gyro_active, y_gyro_active, z_gyro_active });
            }
            else
            {
                if (x_accel_active == 1)
                    this.checkBoxXAccel.Checked = true;
                else
                    this.checkBoxXAccel.Checked = false;

                if (y_accel_active == 1)
                    this.checkBoxYAccel.Checked = true;
                else
                    this.checkBoxYAccel.Checked = false;

                if (z_accel_active == 1)
                    this.checkBoxZAccel.Checked = true;
                else
                    this.checkBoxZAccel.Checked = false;

                if (x_gyro_active == 1)
                    this.checkBoxXGyro.Checked = true;
                else
                    this.checkBoxXGyro.Checked = false;

                if (y_gyro_active == 1)
                    this.checkBoxYGyro.Checked = true;
                else
                    this.checkBoxYGyro.Checked = false;

                if (z_gyro_active == 1)
                    this.checkBoxZGyro.Checked = true;
                else
                    this.checkBoxZGyro.Checked = false;


            }
        }

        private void displayTransmitMode(int broadcast_enabled, int broadcast_rate)
        {
            if (this.broadcastRadio.InvokeRequired)
            {
                displayTransmitModeCallback d = new displayTransmitModeCallback(displayTransmitMode);
                this.Invoke(d, new object[] { broadcast_enabled, broadcast_rate });
            }
            else
            {
                if (broadcast_enabled == 1)
                    this.broadcastRadio.Checked = true;
                else
                    this.listenRadio.Checked = true;

                this.broadcastFrequency.Value = broadcast_rate;
            }
        }

        private void displayAngleEstimate(double angle_estimate)
        {
            if (this.angleEstimateText.InvokeRequired)
            {
                displayAngleEstimateCallback d = new displayAngleEstimateCallback(displayAngleEstimate);
                this.Invoke(d, new object[] { angle_estimate });
            }
            else
            {
                this.angleEstimateText.Text = angle_estimate.ToString();
            }

        }

        private void SaveSettings()
        {
            x_accel = checkBoxXAccel.Checked;
            y_accel = checkBoxYAccel.Checked;
            z_accel = checkBoxZAccel.Checked;
            x_gyro = checkBoxXGyro.Checked;
            y_gyro = checkBoxYGyro.Checked;
            z_gyro = checkBoxZGyro.Checked;

            x_accel_freq = (int)cornerXAccel.Value;
            y_accel_freq = (int)cornerYAccel.Value;
            z_accel_freq = (int)cornerZAccel.Value;
            x_gyro_freq = (int)cornerXGyro.Value;
            y_gyro_freq = (int)cornerYGyro.Value;
            z_gyro_freq = (int)cornerZGyro.Value;
        }
        private void ResetSettings()
        {
            checkBoxXAccel.Checked = x_accel;
            checkBoxYAccel.Checked = y_accel;
            checkBoxZAccel.Checked = z_accel;
            checkBoxXGyro.Checked = x_gyro;
            checkBoxYGyro.Checked = y_gyro;
            checkBoxZGyro.Checked = z_gyro;

            cornerXAccel.Value = x_accel_freq;
            cornerYAccel.Value = y_accel_freq;
            cornerZAccel.Value = z_accel_freq;
            cornerXGyro.Value = x_gyro_freq;
            cornerYGyro.Value = y_gyro_freq;
            cornerZGyro.Value = z_gyro_freq;
        }
        private void refreshGraphs()
        {
            zedGraphControlXAccel.AxisChange();
            zedGraphControlXAccel.Invalidate();
            zedGraphControlYAccel.AxisChange();
            zedGraphControlYAccel.Invalidate();
            zedGraphControlZAccel.AxisChange();
            zedGraphControlZAccel.Invalidate();
            zedGraphControlXGyro.AxisChange();
            zedGraphControlXGyro.Invalidate();
            zedGraphControlYGyro.AxisChange();
            zedGraphControlYGyro.Invalidate();
            zedGraphControlZGyro.AxisChange();
            zedGraphControlZGyro.Invalidate();
        }

        private void saveToolStripMenuItem_Click(object sender, EventArgs e)
        {
            saveFileDialog1.ShowDialog();
        }

        private void timer1_Tick(object sender, EventArgs e)
        {
            time += timer1.Interval / 1000.0;
            if (test > 1.0)
                test = -1.0;

            if (listenRadio.Checked)
            {
                getData();
            }
            
            x_accel_list.Add(time, 0.0001678 * recent_data[0]);
            y_accel_list.Add(time, 0.0001678 * recent_data[1]);
            z_accel_list.Add(time, 0.0001678 * recent_data[2]);
            x_gyro_list.Add(time, 0.02014 * recent_data[3]);
            y_gyro_list.Add(time, 0.02014 * recent_data[4]);
            z_gyro_list.Add(time, 0.02014 * recent_data[5]);

            angle_estimate = angle_estimate + .05 * recent_data[3] * .02014;
            displayAngleEstimate(angle_estimate);
            
            /*
            x_accel_list.Add(time, recent_data[0]);
            y_accel_list.Add(time, recent_data[1]);
            z_accel_list.Add(time, recent_data[2]);
            x_gyro_list.Add(time, recent_data[3]);
            y_gyro_list.Add(time, recent_data[4]);
            z_gyro_list.Add(time, recent_data[5]);
            */
            refreshGraphs();

        }

        private void timer2_Tick(object sender, EventArgs e)
        {
            refreshGraphs();
        }

        #region EVENT HANDLERS
        private void buttonReset_Click(object sender, EventArgs e)
        {
            ResetSettings();
            buttonCommitToRAM.Enabled = false;
        }

        private void buttonCommit_Click(object sender, EventArgs e)
        {
            SaveSettings();
            sendSettingsToDevice();
//            buttonCommitToRAM.Enabled = false;
        }

        private void sendSettingsToDevice()
        {
            setFIRCorners();
            setFIRTaps();
            setActiveChannels();
            setXGyroBias();
            setYGyroBias();
            setZGyroBias();
            setXAccelBias();
            setYAccelBias();
            setZAccelBias();

            if (broadcastRadio.Checked)
                setBroadcastMode();
            else
                setSilentMode();
        }

        private void numericUpDownXGyro_ValueChanged(object sender, EventArgs e)
        {
            int value = (int)cornerXGyro.Value;
            int tmp = value % 10;
            if (tmp != 0 && tmp != 10)
            {
                if (tmp >= 5)
                    cornerXGyro.Value = value - tmp + 10;
                else
                    cornerXGyro.Value = value - tmp;
            }
            buttonCommitToRAM.Enabled = true;
        }

        private void numericUpDownYGyro_ValueChanged(object sender, EventArgs e)
        {
            int tmp = (int)cornerYGyro.Value % 10;
            if (tmp != 0 && tmp != 10)
            {
                if (tmp >= 5)
                    cornerYGyro.Value = cornerYGyro.Value - tmp + 10;
                else
                    cornerYGyro.Value = cornerYGyro.Value - tmp;
            }
        }

        private void numericUpDownZGyro_ValueChanged(object sender, EventArgs e)
        {
            int tmp = (int)cornerZGyro.Value % 10;
            if (tmp != 0 && tmp != 10)
            {
                if (tmp >= 5)
                    cornerZGyro.Value = cornerZGyro.Value - tmp + 10;
                else
                    cornerZGyro.Value = cornerZGyro.Value - tmp;
            }
        }

        private void numericUpDownXAccel_ValueChanged(object sender, EventArgs e)
        {
            int tmp = (int)cornerXAccel.Value % 10;
            if (tmp != 0 && tmp != 10)
            {
                if (tmp >= 5)
                    cornerXAccel.Value = cornerXAccel.Value - tmp + 10;
                else
                    cornerXAccel.Value = cornerXAccel.Value - tmp;
            }
        }

        private void numericUpDownYAccel_ValueChanged(object sender, EventArgs e)
        {
            int tmp = (int)cornerYAccel.Value % 10;
            if (tmp != 0 && tmp != 10)
            {
                if (tmp >= 5)
                    cornerYAccel.Value = cornerYAccel.Value - tmp + 10;
                else
                    cornerYAccel.Value = cornerYAccel.Value - tmp;
            }
        }

        private void numericUpDownZAccel_ValueChanged(object sender, EventArgs e)
        {
            int tmp = (int)cornerZAccel.Value % 10;
            if (tmp != 0 && tmp != 10)
            {
                if (tmp >= 5)
                    cornerZAccel.Value = cornerZAccel.Value - tmp + 10;
                else
                    cornerZAccel.Value = cornerZAccel.Value - tmp;
            }
        }

        private void toolStripButtonConnect_Click(object sender, EventArgs e)
        {
            try
            {
                //set the properties of our SerialPort Object
                serialPort.PortName = toolStripComboBoxPort.SelectedItem.ToString();
                serialPort.BaudRate = int.Parse(toolStripComboBoxBaud.SelectedItem.ToString());
                serialPort.Parity = Parity.None;
                serialPort.DataBits = 8;
                serialPort.StopBits = StopBits.One;
                serialPort.Handshake = Handshake.None;

                //now open the port
                serialPort.Open();

                //display message
                AppendStatusText("\r\n" + serialPort.PortName + " opened at " + serialPort.BaudRate + " baud", Color.Blue);

                getAllConfig();

                serialDisconnectButton.Enabled = true;
                serialConnectButton.Enabled = false;

                startPlottingButton.Enabled = true;

            }
            catch (Exception ex)
            {
                AppendStatusText("\r\nFailed to open " + serialPort.PortName, Color.Red);
                AppendStatusText("\r\n" + ex.ToString(), Color.Red);
            }

        }
        #endregion

        private void handle_packet(byte type, int length, byte[] data)
        {

            switch(type)
            {
                // COMMAND_COMPLETE packet
                case 0xB0:
                    string message;

                    if (data.Length > 0)
                    {
                        switch (data[0])
                        {
                            case 0x80:
                                message = "SET_FIR_CORNERS";
                                break;
                            case 0x81:
                                message = "SET_FIR_TAPS";
                                break;
                            case 0x82:
                                message = "SET_ACTIVE_CHANNELS";
                                break;
                            case 0x83:
                                message = "SET_SILENT_MODE";
                                break;
                            case 0x84:
                                message = "SET_BROADCAST_MODE";
                                break;
                            case 0x85:
                                message = "SET_X_GYRO_BIAS";
                                break;
                            case 0x86:
                                message = "SET_Y_GYRO_BIAS";
                                break;
                            case 0x87:
                                message = "SET_Z_GYRO_BIAS";
                                break;
                            case 0x88:
                                message = "SET_X_ACCEL_BIAS";
                                break;
                            case 0x89:
                                message = "SET_Y_ACCEL_BIAS";
                                break;
                            case 0x8A:
                                message = "SET_Z_ACCEL_BIAS";
                                break;
                            case 0x8B:
                                message = "ZERO_RATE_GYROS";
                                break;
                            case 0x8C:
                                message = "SELF_TEST";
                                break;
                            case 0xA0:
                                message = "WRITE_TO_FLASH";
                                break;
                            default:
                                message = "UNKNOWN_COMMAND";
                                break;
                        }
                        AppendStatusText("\r\n" + message + " complete", Color.Green);
                    }
                    else
                    {
                        AppendStatusText("\r\nUNKNOWN_COMMAND complete", Color.Green);
                    }

                    break;


                // COMMAND_FAILED packet
                case 0xB1:
                    AppendStatusText("\r\nCommand Failed.", Color.Green);

                    break;

                // Bad checksum packet
                case 0xB2:
                    AppendStatusText("\r\nReceived BAD_CHECKSUM", Color.Green);
                    break;
                
                // BAD_DATA_LENGTH packet
                case 0xB3:
                    AppendStatusText("\r\nReceived BAD_DATA_LENGTH", Color.Green);
                    break;

                // UNRECOGNIZED_PACKET
                case 0xB4:
                    AppendStatusText("\r\nReceived UNRECOGNIZED_PACKET", Color.Green);
                    break;

                // Buffer overflow packet
                case 0xB5:
                    AppendStatusText("\r\nReceived BUFFER_OVERFLOW", Color.Green);
                    break;
                
                // TODO: Status Packet
                case 0xB6:

                    break;

                //telemetry packet
                case 0xB7: 
                    time += 0.01;
                    byte act_channels = data[0];
                    int i = 1;
                    if ((act_channels & 0x20) == 0x20)
                    {
                        double tmp = -1 * (double)((data[i] & 0x80) << 8) + (double)(((data[i] & 0x7F) << 8) + data[i + 1]);
                        //z_gyro_list.Add(time, (int)data[i] * Math.Pow(2, 8) + (int)data[i + 1]);
                        //z_gyro_list.Add(time,tmp);
                        recent_data[5] = tmp;
                        i += 2;
                    }
                    if ((act_channels & 0x10) == 0x10)
                    {
                        double tmp = -1 * (double)((data[i] & 0x80) << 8) + (double)(((data[i] & 0x7F) << 8) + data[i + 1]);
                        //y_gyro_list.Add(time, (int)data[i] * Math.Pow(2, 8) + (int)data[i + 1]);
                        //y_gyro_list.Add(time, tmp);
                        recent_data[4] = tmp;
                        i += 2;
                    }
                    if ((act_channels & 0x08) == 0x08)
                    {
                        double tmp = -1 * (double)((data[i] & 0x80) << 8) + (double)(((data[i] & 0x7F) << 8) + data[i + 1]);

                        recent_data[3] = tmp;
                        i += 2;
                    }
                    if ((act_channels & 0x04) == 0x04)
                    {
                        double tmp = -1 * (double)((data[i] & 0x80) << 8) + (double)(((data[i] & 0x7F) << 8) + data[i + 1]);
                       // z_accel_list.Add(time, tmp);
                        i += 2;
                        recent_data[2] = tmp;
                    }
                    if ((act_channels & 0x02) == 0x02)
                    {
                        double tmp = -1 * (double)((data[i] & 0x80) << 8) + (double)(((data[i] & 0x7F) << 8) + data[i + 1]);
                        //y_accel_list.Add(time, (int)data[i] * Math.Pow(2, 8) + (int)data[i + 1]);
                        //y_accel_list.Add(time, tmp);
                        recent_data[1] = tmp;
                        i += 2;
                    }
                    if ((act_channels & 0x01) == 0x01)
                    {
                        double tmp = -1 * (double)((data[i] & 0x80) << 8) + (double)(((data[i] & 0x7F) << 8) + data[i + 1]);
                       // x_accel_list.Add(time, (int)data[i] * Math.Pow(2, 8) + (int)data[i + 1]);
                        //x_accel_list.Add(time, tmp);
                        recent_data[0] = tmp;
                        i += 2;
                    }
                    break;

                // GYRO_BIAS_REPORT packet
                case 0xB8:

                    uint x_gyro_bias = (uint)((data[4] & 0x0FF) << 8) + (uint)(data[5] & 0x0FF);
                    uint y_gyro_bias = (uint)((data[2] & 0x0FF) << 8) + (uint)(data[3] & 0x0FF);
                    uint z_gyro_bias = (uint)((data[0] & 0x0FF) << 8) + (uint)(data[1] & 0x0FF);

                    displayGyroBiases(x_gyro_bias, y_gyro_bias, z_gyro_bias);
                    
                    AppendStatusText("\r\nReceived GYRO_BIAS_REPORT", Color.Green);
                    break;

                // ACCEL_BIAS_REPORT packet
                case 0xB9:
                    uint x_accel_bias = (uint)((data[4] & 0x0FF) << 8) + (uint)(data[5] & 0x0FF);
                    uint y_accel_bias = (uint)((data[2] & 0x0FF) << 8) + (uint)(data[3] & 0x0FF);
                    uint z_accel_bias = (uint)((data[0] & 0x0FF) << 8) + (uint)(data[1] & 0x0FF);

                    displayAccelBiases(x_accel_bias, y_accel_bias, z_accel_bias);

                    AppendStatusText("\r\nReceived ACCEL_BIAS_REPORT", Color.Green);
                    break;

                // FIR_CONFIG_REPORT packet
                case 0xBA:
                    uint x_accel_corner = (uint)((data[2] & 0x0F));
                    uint y_accel_corner = (uint)((data[2] >> 4) & 0x0F);
                    uint z_accel_corner = (uint)((data[1]) & 0x0F);
                    uint x_gyro_corner = (uint)((data[1] >> 4) & 0x0F);
                    uint y_gyro_corner = (uint)((data[0]) & 0x0F);
                    uint z_gyro_corner = (uint)((data[0] >> 4) & 0x0F);

                    displayFilterCorners(x_accel_corner, y_accel_corner, z_accel_corner, x_gyro_corner, y_gyro_corner, z_gyro_corner);

                    AppendStatusText("\r\nReceived FIR_CONFIG_REPORT", Color.Green);
                    break;

                // FIR_TAP_CONFIG_REPORT packet
                case 0xBB:
                    uint x_accel_taps = (uint)((data[1] & 0x03));
                    uint y_accel_taps = (uint)((data[1] >> 2) & 0x03);
                    uint z_accel_taps = (uint)((data[1] >> 4) & 0x03);
                    uint x_gyro_taps = (uint)((data[1] >> 6) & 0x03);
                    uint y_gyro_taps = (uint)((data[0]) & 0x03);
                    uint z_gyro_taps = (uint)((data[0] >> 2) & 0x03);

                    displayFilterTaps(x_accel_taps, y_accel_taps, z_accel_taps, x_gyro_taps, y_gyro_taps, z_gyro_taps);

                    AppendStatusText("\r\nReceived FIR_TAP_CONFIG_REPORT", Color.Green);
                    break;

                // ACTIVE_CHANNEL_REPORT packet
                case 0xBC:
                    int x_accel_active = data[0] & 0x01;
                    int y_accel_active = (data[0] >> 1) & 0x01;
                    int z_accel_active = (data[0] >> 2) & 0x01;
                    int x_gyro_active = (data[0] >> 3) & 0x01;
                    int y_gyro_active = (data[0] >> 4) & 0x01;
                    int z_gyro_active = (data[0] >> 5) & 0x01;

                    displayActiveChannels(x_accel_active, y_accel_active, z_accel_active, x_gyro_active, y_gyro_active, z_gyro_active);

                    AppendStatusText("\r\nReceived ACTIVE_CHANNEL_REPORT", Color.Green);
                    break;
                
                // BROADCAST_MODE_REPORT packet
                case 0xBD:
                    int broadcast_rate = data[0];
                    int broadcast_enabled = data[1] & 0x01;

                    displayTransmitMode(broadcast_enabled, broadcast_rate);

                    AppendStatusText("\r\nReceived BROADCAST_MODE_REPORT", Color.Green);

                    break;

                default:
                    AppendStatusText("\r\nReceived unknown packet", Color.Blue);
                    break;

            }

        }

        private void serialDisconnectButton_Click(object sender, EventArgs e)
        {
            serialPort.Close();

            AppendStatusText("\r\n" + serialPort.PortName + " closed", Color.Blue);
            serialDisconnectButton.Enabled = false;
            serialConnectButton.Enabled = true;

            startPlottingButton.Enabled = false;
            stopPlottingButton.Enabled = false;

            timer1.Stop();
        }

        private void clearSTATUSToolStripMenuItem_Click(object sender, EventArgs e)
        {
            StatusBox.Text = "";
        }

        private void sendPacket(byte packet_type, byte[] data)
        {
            byte[] packet = new byte[20];
            packet[0] = (byte)'s';
            packet[1] = (byte)'n';
            packet[2] = (byte)'p';

            packet[3] = packet_type;
            packet[4] = (byte)data.Length;
            for (int i = 0; i < data.Length; i++)
                packet[i + 5] = data[i];
            
            //create checksum
            int checksum = 0;
            for (int i = 0; i < (data.Length + 5); i++)
            {
                checksum += packet[i];
            }

            packet[data.Length + 5] = (byte)((checksum >> 8) & 0x0FF);
            packet[data.Length + 6] = (byte)((checksum) & 0x0FF);

            serialPort.Write(packet, 0, data.Length + 7);
        }

        private void startPlottingButton_Click(object sender, EventArgs e)
        {
            timer1.Start();

            startPlottingButton.Enabled = false;
            stopPlottingButton.Enabled = true;

            AppendStatusText("\r\nStarted plotting", Color.Blue);

        }

        private void stopPlottingButton_Click(object sender, EventArgs e)
        {
            timer1.Stop();

            startPlottingButton.Enabled = true;
            stopPlottingButton.Enabled = false;

            AppendStatusText("\r\nStopped plotting", Color.Blue);
        }

        private void zeroGyrosButton_Click(object sender, EventArgs e)
        {
            byte[] data = new byte[0];

            sendPacket(0x8B, data);

            AppendStatusText("\r\nSent ZERO_RATE_GYROS packet", Color.Blue);
        }

        private void setFIRCorners()
        {
            byte[] data = new byte[6];

            data[0] = (byte)(cornerZGyro.Value/10 + 1);
            data[1] = (byte)(cornerYGyro.Value/10 + 1);
            data[2] = (byte)(cornerXGyro.Value/10 + 1);
            data[3] = (byte)(cornerZAccel.Value/10 + 1);
            data[4] = (byte)(cornerYAccel.Value/10 + 1);
            data[5] = (byte)(cornerXAccel.Value/10 + 1);
            
            sendPacket(0x80, data);

            AppendStatusText("\r\nSent SET_FIR_CORNERS packet", Color.Blue);
        }

        private void setFIRTaps()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((byte)comboBoxYGyro.SelectedIndex | ((byte)comboBoxZGyro.SelectedIndex << 2));
            data[1] = (byte)((comboBoxXGyro.SelectedIndex << 6) | (comboBoxZAccel.SelectedIndex << 4)  | (comboBoxYAccel.SelectedIndex << 2) | (comboBoxXAccel.SelectedIndex));

            sendPacket(0x81, data);

            AppendStatusText("\r\nSent SET_FIR_TAPS packet", Color.Blue);
        }

        private void setActiveChannels()
        {
            byte[] data = new byte[1];

            int x_accel_enabled;
            int y_accel_enabled;
            int z_accel_enabled;
            int x_gyro_enabled;
            int y_gyro_enabled;
            int z_gyro_enabled;

            if( checkBoxXAccel.Checked )
                x_accel_enabled = 1;
            else
                x_accel_enabled = 0;

            if( checkBoxYAccel.Checked )
                y_accel_enabled = 1;
            else
                y_accel_enabled = 0;

            if( checkBoxZAccel.Checked )
                z_accel_enabled = 1;
            else
                z_accel_enabled = 0;

            if( checkBoxXGyro.Checked )
                x_gyro_enabled = 1;
            else
                x_gyro_enabled = 0;

            if( checkBoxYGyro.Checked )
                y_gyro_enabled = 1;
            else
                y_gyro_enabled = 0;

            if( checkBoxZGyro.Checked )
                z_gyro_enabled = 1;
            else
                z_gyro_enabled = 0;

            data[0] = (byte)(x_accel_enabled | (y_accel_enabled << 1) | (z_accel_enabled << 2) | (x_gyro_enabled << 3) | (y_gyro_enabled << 4) | (z_gyro_enabled << 5));
            
            sendPacket(0x82, data);

            AppendStatusText("\r\nSent SET_ACTIVE_CHANNELS packet", Color.Blue);
        }

        private void setSilentMode()
        {
            byte[] data = new byte[0];

            sendPacket(0x83, data);

            AppendStatusText("\r\nSent SET_SILENT_MODE packet", Color.Blue);
        }

        private void setBroadcastMode()
        {
            byte[] data = new byte[1];

            data[0] = (byte)broadcastFrequency.Value;

            sendPacket(0x84, data);

            AppendStatusText("\r\nSent SET_BROADCAST_MODE packet", Color.Blue);
        }

        private void setXGyroBias()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((int.Parse(biasXGyro.Text) >> 8) & 0x0FF);
            data[1] = (byte)((int.Parse(biasXGyro.Text)) & 0x0FF);

            sendPacket(0x85, data);

            AppendStatusText("\r\nSent SET_X_GYRO_BIAS packet", Color.Blue);
        }

        private void setYGyroBias()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((int.Parse(biasYGyro.Text) >> 8) & 0x0FF);
            data[1] = (byte)((int.Parse(biasYGyro.Text)) & 0x0FF);

            sendPacket(0x86, data);

            AppendStatusText("\r\nSent SET_Y_GYRO_BIAS packet", Color.Blue);
        }

        private void setZGyroBias()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((int.Parse(biasZGyro.Text) >> 8) & 0x0FF);
            data[1] = (byte)((int.Parse(biasZGyro.Text)) & 0x0FF);

            sendPacket(0x87, data);

            AppendStatusText("\r\nSent SET_Z_GYRO_BIAS packet", Color.Blue);
        }

        private void setXAccelBias()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((int.Parse(biasXAccel.Text) >> 8) & 0x0FF);
            data[1] = (byte)((int.Parse(biasXAccel.Text)) & 0x0FF);

            sendPacket(0x88, data);

            AppendStatusText("\r\nSent SET_X_ACCEL_BIAS packet", Color.Blue);
        }

        private void setYAccelBias()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((int.Parse(biasYAccel.Text) >> 8) & 0x0FF);
            data[1] = (byte)((int.Parse(biasYAccel.Text)) & 0x0FF);

            sendPacket(0x89, data);

            AppendStatusText("\r\nSent SET_Y_ACCEL_BIAS packet", Color.Blue);
        }

        private void setZAccelBias()
        {
            byte[] data = new byte[2];

            data[0] = (byte)((int.Parse(biasZAccel.Text) >> 8) & 0x0FF);
            data[1] = (byte)((int.Parse(biasZAccel.Text)) & 0x0FF);

            sendPacket(0x8A, data);

            AppendStatusText("\r\nSent SET_Z_ACCEL_BIAS packet", Color.Blue);
        }

        private void writeToFlash()
        {
            byte[] data = new byte[0];

            sendPacket(0xA0, data);

            AppendStatusText("\r\nSent WRITE_TO_FLASH packet", Color.Blue);
        }

        private void getData()
        {
            byte[] data = new byte[0];

            sendPacket(0x01, data);

//            AppendStatusText("\r\nSent GET_DATA packet", Color.Blue);
        }

        private void getGyroBias()
        {
            byte[] data = new byte[0];

            sendPacket(0x02, data);

            AppendStatusText("\r\nSent GET_GYRO_BIAS packet", Color.Blue);
        }

        private void getAccelBias()
        {
            byte[] data = new byte[0];

            sendPacket(0x03, data);

            AppendStatusText("\r\nSent GET_ACCEL_BIAS packet", Color.Blue);
        }

        private void getFIRConfig()
        {
            byte[] data = new byte[0];

            sendPacket(0x04, data);

            AppendStatusText("\r\nSent GET_FIR_CONFIG packet", Color.Blue);
        }

        private void getFIRTapConfig()
        {
            byte[] data = new byte[0];

            sendPacket(0x05, data);

            AppendStatusText("\r\nSent GET_FIR_TAP_CONFIG packet", Color.Blue);
        }

        private void getActiveChannels()
        {
            byte[] data = new byte[0];

            sendPacket(0x06, data);

            AppendStatusText("\r\nSent GET_ACTIVE_CHANNELS packet", Color.Blue);
        }

        private void getTransmitMode()
        {
            byte[] data = new byte[0];

            sendPacket(0x07, data);

            AppendStatusText("\r\nSent GET_BROADCAST_MODE packet", Color.Blue);
        }

        private void buttonLoad_Click(object sender, EventArgs e)
        {
            getAllConfig();
        }

        private void getAllConfig()
        {
            // Send commands to IMU to retrieve configuration parameters
            getGyroBias();
            getActiveChannels();
            getFIRTapConfig();
            getFIRConfig();
            getAccelBias();
            getTransmitMode();
        }

        private void buttonCommitToFlash_Click(object sender, EventArgs e)
        {
            writeToFlash();
        }

        private void angleEstimateReset_Click(object sender, EventArgs e)
        {
            angle_estimate = 0.0;

            displayAngleEstimate( angle_estimate );
        }

    }
}